<?php

namespace App\Http\Controllers;

use App\Models\CasteMaster;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class CasteMasterController extends Controller
{
    /**
     * Display a listing
     */
    public function index(Request $request)
    {
        try {
            $perPage = $request->get('per_page', 25);
            $search = $request->get('search', '');
            $withTrashed = $request->get('with_trashed', false);

            $query = CasteMaster::orderBy('Caste_Id', 'desc');

            // Include soft deleted if requested
            if ($withTrashed) {
                $query->withTrashed();
            }

            // Search filter
            if ($search) {
                $query->where(function ($q) use ($search) {
                    $q->where('caste_name', 'like', "%{$search}%")
                        ->orWhere('Caste_Id', 'like', "%{$search}%");
                });
            }

            // Pagination or all
            if ($perPage === 'all') {
                $data = $query->get();
            } else {
                $data = $query->paginate((int) $perPage);
            }

            return response()->json([
                'status' => true,
                'message' => 'Caste data fetched successfully',
                'data' => $data
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'status' => false,
                'message' => 'Something went wrong',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    public function all()
    {
        try {
            return response()->json([
                'status' => true,
                'data' => CasteMaster::latest()->get()
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'status' => false,
                'message' => $e->getMessage()
            ], 500);
        }
    }


    /**
     * Store
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'caste_name' => 'required|string|max:50'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => false,
                'message' => $validator->errors()
            ], 422);
        }

        $caste = CasteMaster::create([
            'caste_name' => $request->caste_name
        ]);

        return response()->json([
            'status' => true,
            'message' => 'Caste created successfully',
            'data' => $caste
        ]);
    }

    /**
     * Show
     */
    public function show($id)
    {
        $caste = CasteMaster::find($id);

        if (!$caste) {
            return response()->json([
                'status' => false,
                'message' => 'Caste not found'
            ], 404);
        }

        return response()->json([
            'status' => true,
            'data' => $caste
        ]);
    }

    /**
     * Update
     */
    public function update(Request $request, $id)
    {
        $caste = CasteMaster::find($id);

        if (!$caste) {
            return response()->json([
                'status' => false,
                'message' => 'Caste not found'
            ], 404);
        }

        $validator = Validator::make($request->all(), [
            'caste_name' => 'required|string|max:50'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => false,
                'message' => $validator->errors()
            ], 422);
        }

        $caste->update([
            'caste_name' => $request->caste_name
        ]);

        return response()->json([
            'status' => true,
            'message' => 'Caste updated successfully',
            'data' => $caste
        ]);
    }

    /**
     * Delete (Soft Delete)
     */
    public function destroy($id)
    {
        $caste = CasteMaster::find($id);

        if (!$caste) {
            return response()->json([
                'status' => false,
                'message' => 'Caste not found'
            ], 404);
        }

        $caste->delete();

        return response()->json([
            'status' => true,
            'message' => 'Caste deleted successfully'
        ]);
    }
}
