<?php

namespace App\Http\Controllers;

use App\Models\CccMaster;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class CccMasterController extends Controller
{
    public function index(Request $request)
    {
        try {
            $query = CccMaster::with(['ward', 'ward.zone', 'ward.zone.organisation'])
                        ->active();

            if ($request->has('wrd_id')) {
                $query->byWard($request->wrd_id);
            }

            if ($request->has('zone_id')) {
                $query->whereHas('ward', function ($q) use ($request) {
                    $q->where('Zone_Id', $request->zone_id);
                });
            }

            if ($request->has('search')) {
                $query->search($request->search);
            }

            $perPage = $request->get('per_page', 15);
            $cccs = $query->paginate($perPage);

            return response()->json([
                'success' => true,
                'data' => $cccs,
                'message' => 'CCCs retrieved successfully'
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to retrieve CCCs',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    public function store(Request $request)
    {
        try {

            $validator = Validator::make(
                $request->all(),
                CccMaster::createRules(),
                CccMaster::validationMessages()
            );

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'errors' => $validator->errors()
                ], 422);
            }

            $ccc = CccMaster::create($request->all());

            return response()->json([
                'success' => true,
                'message' => 'CCC created successfully',
                'data' => $ccc->load(['ward', 'ward.zone'])
            ], 201);

        } catch (\Exception $e) {

            return response()->json([
                'success' => false,
                'message' => 'Failed to create CCC',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    public function show($id)
    {
        try {

            $ccc = CccMaster::with(['ward', 'ward.zone'])->find($id);

            if (!$ccc) {
                return response()->json([
                    'success' => false,
                    'message' => 'CCC not found'
                ], 404);
            }

            return response()->json([
                'success' => true,
                'data' => $ccc
            ]);

        } catch (\Exception $e) {

            return response()->json([
                'success' => false,
                'message' => 'Failed to retrieve CCC',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    public function update(Request $request, $id)
    {
        try {

            $ccc = CccMaster::find($id);

            if (!$ccc) {
                return response()->json([
                    'success' => false,
                    'message' => 'CCC not found'
                ], 404);
            }

            $validator = Validator::make(
                $request->all(),
                CccMaster::updateRules($id),
                CccMaster::validationMessages()
            );

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'errors' => $validator->errors()
                ], 422);
            }

            $ccc->update($request->all());

            return response()->json([
                'success' => true,
                'message' => 'CCC updated successfully',
                'data' => $ccc->load(['ward', 'ward.zone'])
            ]);

        } catch (\Exception $e) {

            return response()->json([
                'success' => false,
                'message' => 'Failed to update CCC',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    public function destroy($id)
    {
        try {

            $ccc = CccMaster::find($id);

            if (!$ccc) {
                return response()->json([
                    'success' => false,
                    'message' => 'CCC not found'
                ], 404);
            }

            $ccc->deactivate();

            return response()->json([
                'success' => true,
                'message' => 'CCC deactivated successfully'
            ]);

        } catch (\Exception $e) {

            return response()->json([
                'success' => false,
                'message' => 'Failed to deactivate CCC',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    public function restore($id)
    {
        try {

            $ccc = CccMaster::withTrashed()->find($id);

            if (!$ccc) {
                return response()->json([
                    'success' => false,
                    'message' => 'CCC not found'
                ], 404);
            }

            $ccc->activate();

            return response()->json([
                'success' => true,
                'message' => 'CCC activated successfully',
                'data' => $ccc->load(['ward', 'ward.zone'])
            ]);

        } catch (\Exception $e) {

            return response()->json([
                'success' => false,
                'message' => 'Failed to activate CCC',
                'error' => $e->getMessage()
            ], 500);
        }
    }
}
