<?php

namespace App\Http\Controllers;

use App\Http\Controllers\Controller;
use App\Models\DebtTypeMaster;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class DebtTypeMasterController extends Controller
{
   public function index(Request $request)
{
    try {

        $perPage = $request->get('per_page', 25);
        $search = $request->get('search', '');
        $withTrashed = $request->get('with_trashed', false);

        $query = DebtTypeMaster::where('Is_Active', true)
                    ->orderBy('DT_Id', 'desc');

        // Include soft deleted if requested
        if ($withTrashed) {
            $query->withTrashed();
        }

        // Search filter
        if ($search) {
            $query->where(function ($q) use ($search) {
                $q->where('DT_Name', 'like', "%{$search}%")
                  ->orWhere('DT_Code', 'like', "%{$search}%")
                  ->orWhere('DT_Id', 'like', "%{$search}%");
            });
        }

        // Pagination or all
        if ($perPage === 'all') {
            $data = $query->get();
        } else {
            $data = $query->paginate((int) $perPage);
        }

        return response()->json([
            'success' => true,
            'data' => $data,
            'options' => [
                'dt_cate_options' => DebtTypeMaster::getDtCateOptions(),
                'dt_lia_type_options' => DebtTypeMaster::getDtLiaTypeOptions()
            ]
        ]);

    } catch (\Exception $e) {
        return response()->json([
            'success' => false,
            'message' => $e->getMessage()
        ], 500);
    }
}
public function all()
{
    try {

        $data = DebtTypeMaster::where('Is_Active', true)
                    ->orderBy('DT_Id', 'desc')
                    ->get();

        return response()->json([
            'success' => true,
            'data' => $data,
            'options' => [
                'dt_cate_options' => DebtTypeMaster::getDtCateOptions(),
                'dt_lia_type_options' => DebtTypeMaster::getDtLiaTypeOptions()
            ]
        ]);

    } catch (\Exception $e) {
        return response()->json([
            'success' => false,
            'message' => $e->getMessage()
        ], 500);
    }
}


    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'DT_Code' => 'nullable|string|max:20|unique:debt_type_master,DT_Code',
            'DT_Name' => 'required|string|max:50',
            'DT_Cate' => 'nullable|in:Long Term,Short Term',
            'DT_Lia_type' => 'nullable|in:Payable,Deposit,Loan',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors()
            ], 422);
        }

        $debtType = DebtTypeMaster::create($request->all());

        return response()->json([
            'success' => true,
            'message' => 'Debt type created successfully',
            'data' => $debtType
        ], 201);
    }

    public function show($id)
    {
        $debtType = DebtTypeMaster::find($id);

        if (!$debtType) {
            return response()->json([
                'success' => false,
                'message' => 'Debt type not found'
            ], 404);
        }

        return response()->json([
            'success' => true,
            'data' => $debtType,
            'options' => [
                'dt_cate_options' => DebtTypeMaster::getDtCateOptions(),
                'dt_lia_type_options' => DebtTypeMaster::getDtLiaTypeOptions()
            ]
        ]);
    }

    public function update(Request $request, $id)
    {
        $debtType = DebtTypeMaster::find($id);

        if (!$debtType) {
            return response()->json([
                'success' => false,
                'message' => 'Debt type not found'
            ], 404);
        }

        $validator = Validator::make($request->all(), [
            'DT_Code' => 'nullable|string|max:20|unique:debt_type_master,DT_Code,' . $id . ',DT_id',
            'DT_Name' => 'required|string|max:50',
            'DT_Cate' => 'nullable|in:Long Term,Short Term',
            'DT_Lia_type' => 'nullable|in:Payable,Deposit,Loan',
            'Is_Active' => 'boolean'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors()
            ], 422);
        }

        $debtType->update($request->all());

        return response()->json([
            'success' => true,
            'message' => 'Debt type updated successfully',
            'data' => $debtType
        ]);
    }

    public function destroy($id)
    {
        $debtType = DebtTypeMaster::find($id);

        if (!$debtType) {
            return response()->json([
                'success' => false,
                'message' => 'Debt type not found'
            ], 404);
        }

        $debtType->update(['Is_Active' => false]);

        return response()->json([
            'success' => true,
            'message' => 'Debt type deactivated successfully'
        ]);
    }

    public function restore($id)
    {
        $debtType = DebtTypeMaster::find($id);

        if (!$debtType) {
            return response()->json([
                'success' => false,
                'message' => 'Debt type not found'
            ], 404);
        }

        $debtType->update(['Is_Active' => true]);

        return response()->json([
            'success' => true,
            'message' => 'Debt type activated successfully',
            'data' => $debtType
        ]);
    }

    public function getOptions()
    {
        return response()->json([
            'success' => true,
            'data' => [
                'dt_cate_options' => DebtTypeMaster::getDtCateOptions(),
                'dt_lia_type_options' => DebtTypeMaster::getDtLiaTypeOptions()
            ]
        ]);
    }
}
