<?php

namespace App\Http\Controllers;

use App\Models\DeptMenuMap;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use App\Models\Department; // Dept model
use App\Models\FunctionType; // Function/Menu model

class DeptMenuMapController extends Controller
{
    // =============================
    // LIST ALL
    // =============================
public function index()
    {
        // Eager load the department relation to avoid N+1 queries
        $items = DeptMenuMap::with('department')->get();

        $data = $items->map(function ($item) {
            // Split comma-separated menu IDs safely
            $menuIds = $item->Fnc_Id
                ? array_map('trim', explode(',', $item->Fnc_Id))
                : [];

            // Prepare department data (null if no relation)
            $departmentData = null;
            if ($item->department) {
                $departmentData = [
                    'Dpt_Id'    => $item->department->Dpt_Id,
                    'Dpt_Name'  => $item->department->Dpt_Name,
                    'GrCod'     => $item->department->GrCod,
                    'UpGrp'     => $item->department->UpGrp,
                    'GrSrn'     => $item->department->GrSrn,
                    'GrStt'     => $item->department->GrStt,
                    'isvisible' => $item->department->isvisible,
                    'Dpt_Desc'  => $item->department->Dpt_Desc ?? null,
                ];
            }

            return [
                'DMM_Id'     => $item->DMM_Id,
                'Dpt_Id'     => $item->Dpt_Id,           // ← this is the fix you wanted
                'department' => $departmentData,
                'Fnc_Id'     => $menuIds,
                'menus'      => FunctionType::whereIn('Fnc_Id', $menuIds)
                    ->get([
                        'Fnc_Id',
                        'Fnc_Title',
                        'Fnc_Desc',
                        'GrCod',
                        'UpGrp',
                        'GrSrn',
                        'GrStt',
                        'isvisible',
                        'Description',
                        'Menu_Links',
                        'created_at',
                        'updated_at',
                        'deleted_at',
                    ]),
                'Status'     => $item->Status,
                'created_at' => $item->created_at,
                'updated_at' => $item->updated_at,
            ];
        });

        return response()->json([
            'success' => true,
            'data'    => $data
        ]);
    }
    // =============================
    // STORE
    // =============================
public function store(Request $request)
{
    $validator = Validator::make($request->all(), [
        'Dpt_Id'   => 'required|exists:dept_master,Dpt_id',
        'Fnc_Id'   => 'required|array',
        'Fnc_Id.*' => 'exists:function_types,Fnc_Id'
    ]);

    if ($validator->fails()) {
        return response()->json([
            'success' => false,
            'errors'  => $validator->errors()
        ], 422);
    }

    $mapping = DeptMenuMap::create([
        'Dpt_Id' => $request->Dpt_Id,                    // ← fixed
        'Fnc_Id' => implode(',', $request->Fnc_Id),
        'Status' => 1
    ]);

    return response()->json([
        'success' => true,
        'message' => 'Mapping created successfully',
        'data'    => $mapping
    ]);
}

    // =============================
    // SHOW
    // =============================
public function show($id)
{
    // Add ->with('department')
    $item = DeptMenuMap::with('department')->find($id);

    if (!$item) {
        return response()->json([
            'success' => false,
            'message' => 'Record not found'
        ], 404);
    }

    $menuIds = $item->Fnc_Id ? explode(',', $item->Fnc_Id) : [];

    $data = [
        'DMM_Id'     => $item->DMM_Id,
        'Dpt_Id'     => $item->Dpt_Id,
        'department' => $item->department
            ? [
                'Dpt_Id'   => $item->department->Dpt_id,
                'Dpt_Name' => $item->department->Dpt_Name,
                'GrCod'    => $item->department->GrCod,
                'UpGrp'    => $item->department->UpGrp,
                'GrSrn'    => $item->department->GrSrn,
                'GrStt'    => $item->department->GrStt,
                'isvisible'=> $item->department->isvisible,
                'Dpt_Desc' => $item->department->Dpt_Desc ?? null,
              ]
            : null,
        'Fnc_Id'     => $menuIds,
        'menus'      => FunctionType::whereIn('Fnc_Id', $menuIds)->get(),
        'Status'     => $item->Status,
        'created_at' => $item->created_at,
        'updated_at' => $item->updated_at
    ];

    return response()->json([
        'success' => true,
        'data'    => $data
    ]);
}

    // =============================
    // UPDATE
    // =============================
public function update(Request $request, $id)
{
    $mapping = DeptMenuMap::find($id);

    if (!$mapping) {
        return response()->json([
            'success' => false,
            'message' => 'Record not found'
        ], 404);
    }

    $validator = Validator::make($request->all(), [
        'Dpt_Id'   => 'sometimes|required|exists:dept_master,Dpt_id',
        'Fnc_Id'   => 'sometimes|required|array',
        'Fnc_Id.*' => 'sometimes|exists:function_types,Fnc_Id',
        'Status'   => 'sometimes|in:0,1', // or boolean if you prefer
    ]);

    if ($validator->fails()) {
        return response()->json([
            'success' => false,
            'errors'  => $validator->errors()
        ], 422);
    }

    $updateData = [];

    if ($request->has('Dpt_Id')) {
        $updateData['Dpt_Id'] = $request->Dpt_Id;
    }

    if ($request->has('Fnc_Id')) {
        $updateData['Fnc_Id'] = implode(',', $request->Fnc_Id);
    }

    if ($request->has('Status')) {
        $updateData['Status'] = $request->Status;
    }

    // Only update if there is something to update
    if (!empty($updateData)) {
        $mapping->update($updateData);
    }

    // Reload with relationship for response
    $mapping->load('department');

    $menuIds = $mapping->Fnc_Id ? array_filter(explode(',', $mapping->Fnc_Id)) : [];

    $responseData = [
        'DMM_Id'     => $mapping->DMM_Id,
        'Dpt_Id'     => $mapping->Dpt_Id,
        'department' => $mapping->department ? [
            'Dpt_Id'    => $mapping->department->Dpt_id,
            'Dpt_Name'  => $mapping->department->Dpt_Name,
            // ... add other fields you need
        ] : null,
        'Fnc_Id'     => $menuIds,
        'menus'      => $menuIds ? FunctionType::whereIn('Fnc_Id', $menuIds)->get() : [],
        'Status'     => $mapping->Status,
        'created_at' => $mapping->created_at,
        'updated_at' => $mapping->updated_at,
    ];

    return response()->json([
        'success' => true,
        'message' => 'Updated successfully',
        'data'    => $responseData
    ]);
}

    // =============================
    // DELETE
    // =============================
    public function destroy($id)
    {
        $mapping = DeptMenuMap::find($id);

        if (!$mapping) {
            return response()->json([
                'success' => false,
                'message' => 'Record not found'
            ], 404);
        }

        $mapping->delete();

        return response()->json([
            'success' => true,
            'message' => 'Deleted successfully'
        ]);
    }
}
