<?php

namespace App\Http\Controllers;

use App\Models\DesignationMaster;
use Illuminate\Http\Request;
use Illuminate\Validation\ValidationException;
use Illuminate\Support\Facades\Log;
use Throwable;

class DesignationMasterController extends Controller
{
    // GET all
    public function index(Request $request)
    {
        try {
            $queryValue = $request->input('query');
            $isvisible  = $request->input('isvisible');

            $query = DesignationMaster::query();

            // Search filter
            if ($queryValue) {
                $query->where('UpGrp', 'like', $queryValue . '%');
            }

            // isvisible filter
            if ($isvisible !== null) {
                $query->where('isvisible', $isvisible);
            }

            $query->orderBy('Dsg_id', 'desc'); // change column if needed

            // Pagination logic
            $perPage = $request->get('per_page', 25);

            if ($perPage === 'all') {
                $data = $query->get();
            } else {
                $data = $query->paginate((int) $perPage);
            }

            return response()->json([
                'status'  => 'success',
                'message' => 'Designation records fetched successfully',
                'data'    => $data
            ]);
        } catch (Throwable $e) {
            Log::error('Fetch Error: ' . $e->getMessage());

            return response()->json([
                'status'  => 'error',
                'message' => 'Unable to fetch designation records',
                'error'   => $e->getMessage()
            ], 500);
        }
    }
    public function all()
    {
        try {
            $data = DesignationMaster::orderBy('Dsg_id', 'desc')->get();

            return response()->json([
                'status'  => 'success',
                'message' => 'Designation records fetched successfully',
                'data'    => $data
            ]);
        } catch (Throwable $e) {
            Log::error('Fetch All Error: ' . $e->getMessage());

            return response()->json([
                'status'  => 'error',
                'message' => 'Unable to fetch designation records'
            ], 500);
        }
    }


    // GET one
    public function show($id)
    {
        try {
            $data = DesignationMaster::find($id);

            if (!$data) {
                return response()->json([
                    'status'  => 'error',
                    'message' => 'Designation record not found'
                ], 404);
            }

            return response()->json([
                'status'  => 'success',
                'message' => 'Designation record fetched successfully',
                'data'    => $data
            ]);
        } catch (Throwable $e) {
            Log::error('Show Error: ' . $e->getMessage());
            return response()->json([
                'status'  => 'error',
                'message' => 'Unable to fetch designation record',
                'error'   => $e->getMessage()
            ], 500);
        }
    }

    // CREATE
    public function store(Request $request)
    {
        try {
            $request->validate([
                'Dsg_Title' => 'nullable|string|max:100',
                'Dsg_Desc'  => 'nullable|string|max:255',
                'Dsg_EName' => 'nullable|string|max:100',
                'UpGrp'     => 'nullable|string|max:30',
                'GrCod'     => 'nullable|string|max:30',
                'GrSrn'     => 'nullable|string|max:30',
                'GrStt'     => 'nullable|string|max:30',
                'isvisible' => 'nullable|integer|in:0,1',
            ]);

            // Default GrStt
            $grstt = $request->input('GrStt', '1');

            // Auto-increment GrSrn based on GrStt
            $grsrn = $request->input('GrSrn')
                ?? (DesignationMaster::where('GrStt', $grstt)->count() + 1);

            $prefix = str_pad($grsrn, 2, '0', STR_PAD_LEFT);
            $code   = $prefix . '000000';

            $data = [
                'Dsg_Title' => $request->input('Dsg_Title'),
                'Dsg_Desc'  => $request->input('Dsg_Desc'),
                'Dsg_EName' => $request->input('Dsg_EName'),
                'UpGrp'     => $request->input('UpGrp', $code),
                'GrCod'     => $request->input('GrCod', $code),
                'GrSrn'     => (string)$grsrn,
                'GrStt'     => $grstt,
                'isvisible' => $request->input('isvisible', 1),
            ];

            $menu = DesignationMaster::create($data);

            return response()->json([
                'status'  => 'success',
                'message' => 'Designation record created successfully',
                'data'    => $menu
            ], 201);
        } catch (ValidationException $e) {
            return response()->json([
                'status'  => 'error',
                'message' => 'Validation failed',
                'errors'  => $e->errors()
            ], 422);
        } catch (Throwable $e) {
            Log::error('Create Error: ' . $e->getMessage());
            return response()->json([
                'status'  => 'error',
                'message' => 'Unable to create designation record',
                'error'   => $e->getMessage()
            ], 500);
        }
    }

    // UPDATE
    public function update(Request $request, $id)
    {
        try {
            $menu = DesignationMaster::find($id);

            if (!$menu) {
                return response()->json([
                    'status'  => 'error',
                    'message' => 'Designation record not found'
                ], 404);
            }

            $data = $request->validate([
                'Dsg_Title' => 'sometimes|nullable|string|max:100',
                'Dsg_Desc'  => 'sometimes|nullable|string|max:255',
                'Dsg_EName' => 'sometimes|nullable|string|max:100',
                'UpGrp'     => 'sometimes|nullable|string|max:30',
                'GrCod'     => 'sometimes|nullable|string|max:30',
                'GrSrn'     => 'sometimes|nullable|string|max:30',
                'GrStt'     => 'sometimes|nullable|string|max:30',
                'isvisible' => 'sometimes|nullable|integer|in:0,1',
            ]);

            $menu->update($data);

            return response()->json([
                'status'  => 'success',
                'message' => 'Designation record updated successfully',
                'data'    => $menu
            ]);
        } catch (ValidationException $e) {
            return response()->json([
                'status'  => 'error',
                'message' => 'Validation failed',
                'errors'  => $e->errors()
            ], 422);
        } catch (Throwable $e) {
            Log::error('Update Error: ' . $e->getMessage());
            return response()->json([
                'status'  => 'error',
                'message' => 'Unable to update designation record',
                'error'   => $e->getMessage()
            ], 500);
        }
    }

    // TOGGLE VISIBILITY
    public function toggleVisibility($id)
    {
        try {
            $menu = DesignationMaster::find($id);

            if (!$menu) {
                return response()->json([
                    'status'  => 'error',
                    'message' => 'Designation record not found'
                ], 404);
            }

            $menu->update([
                'isvisible' => $menu->isvisible == 1 ? 0 : 1
            ]);

            $status = $menu->isvisible == 1 ? 'Active' : 'Inactive';

            return response()->json([
                'status'  => 'success',
                'message' => 'Visibility toggled successfully',
                'data'    => [
                    'Dsg_Id'    => $menu->Dsg_Id,
                    'isvisible' => $menu->isvisible,
                    'status'    => $status
                ]
            ]);
        } catch (Throwable $e) {
            Log::error('Toggle Visibility Error: ' . $e->getMessage());
            return response()->json([
                'status'  => 'error',
                'message' => 'Unable to toggle visibility',
                'error'   => $e->getMessage()
            ], 500);
        }
    }

    // DELETE
    public function destroy($id)
    {
        try {
            $menu = DesignationMaster::find($id);

            if (!$menu) {
                return response()->json([
                    'status'  => 'error',
                    'message' => 'Designation record not found'
                ], 404);
            }

            $menu->delete();

            return response()->json([
                'status'  => 'success',
                'message' => 'Designation record deleted successfully'
            ]);
        } catch (Throwable $e) {
            Log::error('Delete Error: ' . $e->getMessage());
            return response()->json([
                'status'  => 'error',
                'message' => 'Unable to delete designation record',
                'error'   => $e->getMessage()
            ], 500);
        }
    }
}
