<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\EmpCatMaster;
use Illuminate\Http\JsonResponse;

class EmpCatMasterController extends Controller
{

    /**
     * List with Search + Pagination
     */
    public function index(Request $request): JsonResponse
    {
        try {

            $perPage = $request->get('per_page', 15);
            $search = $request->get('search', '');

            $query = EmpCatMaster::query();

            if (!empty($search)) {
                $query->where(function ($q) use ($search) {
                    $q->where('ECt_Name', 'like', "%{$search}%")
                      ->orWhere('ECt_Code', 'like', "%{$search}%");
                });
            }

            $data = $query->orderBy('ECt_Id', 'desc')->paginate($perPage);

            return response()->json([
                'status' => true,
                'message' => 'Emp Category List',
                'data' => $data
            ]);

        } catch (\Exception $e) {

            return response()->json([
                'status' => false,
                'message' => $e->getMessage()
            ], 500);
        }
    }


    /**
     * Store
     */
    public function store(Request $request): JsonResponse
    {
        try {

            $request->validate([
                'ECt_Name' => 'required|string|max:50',
                'ECt_Code' => 'nullable|string|max:20',
            ]);

            $data = EmpCatMaster::create([
                'ECt_Code' => $request->ECt_Code,
                'ECt_Name' => $request->ECt_Name,
                'Is_Active' => $request->Is_Active ?? 1,
            ]);

            return response()->json([
                'status' => true,
                'message' => 'Emp Category Created Successfully',
                'data' => $data
            ]);

        } catch (\Exception $e) {

            return response()->json([
                'status' => false,
                'message' => $e->getMessage()
            ], 500);
        }
    }


    /**
     * Show Single
     */
    public function show($id): JsonResponse
    {
        try {

            $data = EmpCatMaster::find($id);

            if (!$data) {
                return response()->json([
                    'status' => false,
                    'message' => 'Data Not Found'
                ]);
            }

            return response()->json([
                'status' => true,
                'data' => $data
            ]);

        } catch (\Exception $e) {

            return response()->json([
                'status' => false,
                'message' => $e->getMessage()
            ], 500);
        }
    }


    /**
     * Update
     */
    public function update(Request $request, $id): JsonResponse
    {
        try {

            $data = EmpCatMaster::find($id);

            if (!$data) {
                return response()->json([
                    'status' => false,
                    'message' => 'Data Not Found'
                ]);
            }

            $request->validate([
                'ECt_Name' => 'required|string|max:50',
                'ECt_Code' => 'nullable|string|max:20',
            ]);

            $data->update([
                'ECt_Code' => $request->ECt_Code,
                'ECt_Name' => $request->ECt_Name,
                'Is_Active' => $request->Is_Active ?? $data->Is_Active,
            ]);

            return response()->json([
                'status' => true,
                'message' => 'Emp Category Updated Successfully',
                'data' => $data
            ]);

        } catch (\Exception $e) {

            return response()->json([
                'status' => false,
                'message' => $e->getMessage()
            ], 500);
        }
    }


    /**
     * Delete (Soft Delete)
     */
    public function destroy($id): JsonResponse
    {
        try {

            $data = EmpCatMaster::find($id);

            if (!$data) {
                return response()->json([
                    'status' => false,
                    'message' => 'Data Not Found'
                ]);
            }

            $data->delete();

            return response()->json([
                'status' => true,
                'message' => 'Emp Category Deleted Successfully'
            ]);

        } catch (\Exception $e) {

            return response()->json([
                'status' => false,
                'message' => $e->getMessage()
            ], 500);
        }
    }


    /**
     * Get All Active Data
     */
    public function all(): JsonResponse
    {
        try {

            $data = EmpCatMaster::where('Is_Active', 1)
                        ->orderBy('ECt_Name')
                        ->get();

            return response()->json([
                'status' => true,
                'data' => $data
            ]);

        } catch (\Exception $e) {

            return response()->json([
                'status' => false,
                'message' => $e->getMessage()
            ], 500);
        }
    }


    /**
     * Change Status
     */
    public function changeStatus($id): JsonResponse
    {
        try {

            $data = EmpCatMaster::find($id);

            if (!$data) {
                return response()->json([
                    'status' => false,
                    'message' => 'Data Not Found'
                ]);
            }

            $data->Is_Active = !$data->Is_Active;
            $data->save();

            return response()->json([
                'status' => true,
                'message' => 'Status Changed Successfully',
                'data' => $data
            ]);

        } catch (\Exception $e) {

            return response()->json([
                'status' => false,
                'message' => $e->getMessage()
            ], 500);
        }
    }
}
