<?php

namespace App\Http\Controllers;

use App\Models\EmpEduMaster;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;

class EmpEduMasterController extends Controller
{
    /**
     * List with Search & Pagination
     */
    public function index(Request $request): JsonResponse
    {
        try {

            $perPage = $request->get('per_page', 15);
            $search = $request->get('search', '');

            $query = EmpEduMaster::query();

            // Search
            if (!empty($search)) {
                $query->where(function ($q) use ($search) {
                    $q->where('EEd_Code', 'like', "%{$search}%")
                        ->orWhere('EEd_Name', 'like', "%{$search}%");
                });
            }

            $data = $query->orderBy('EEd_Id', 'desc')
                ->paginate($perPage);

            return response()->json([
                'success' => true,
                'message' => 'Employee Education list fetched successfully',
                'data' => $data
            ]);
        } catch (\Exception $e) {

            return response()->json([
                'success' => false,
                'message' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Store
     */
    public function store(Request $request): JsonResponse
    {
        $validated = $request->validate([
            'EEd_Code' => 'nullable|string|max:20',
            'EEd_Name' => 'required|string|max:50',
            'Is_Active' => 'boolean',
            'Is_Mandatory' => 'in:Y,N',
            'Is_Technical' => 'in:Y,N',
            'Is_Professional' => 'in:Y,N',
            'Is_Promotional' => 'in:Y,N',
            'Rec_Valid' => 'in:Y,N'
        ]);

        $data = EmpEduMaster::create($validated);

        return response()->json([
            'success' => true,
            'message' => 'Employee Education created successfully',
            'data' => $data
        ]);
    }

    /**
     * Show Single
     */
    public function show($id): JsonResponse
    {
        $data = EmpEduMaster::find($id);

        if (!$data) {
            return response()->json([
                'success' => false,
                'message' => 'Record not found'
            ], 404);
        }

        return response()->json([
            'success' => true,
            'data' => $data
        ]);
    }

    /**
     * Update
     */
    public function update(Request $request, $id): JsonResponse
    {
        $data = EmpEduMaster::find($id);

        if (!$data) {
            return response()->json([
                'success' => false,
                'message' => 'Record not found'
            ], 404);
        }

        $validated = $request->validate([
            'EEd_Code' => 'nullable|string|max:20',
            'EEd_Name' => 'required|string|max:50',
            'Is_Active' => 'boolean',
            'Is_Mandatory' => 'in:Y,N',
            'Is_Technical' => 'in:Y,N',
            'Is_Professional' => 'in:Y,N',
            'Is_Promotional' => 'in:Y,N',
            'Rec_Valid' => 'in:Y,N'
        ]);

        $data->update($validated);

        return response()->json([
            'success' => true,
            'message' => 'Employee Education updated successfully',
            'data' => $data
        ]);
    }

    /**
     * Delete (Soft Delete)
     */
    public function destroy($id): JsonResponse
    {
        $data = EmpEduMaster::find($id);

        if (!$data) {
            return response()->json([
                'success' => false,
                'message' => 'Record not found'
            ], 404);
        }

        $data->delete();

        return response()->json([
            'success' => true,
            'message' => 'Employee Education deleted successfully'
        ]);
    }

    /**
     * All Active Records
     */
    public function all(): JsonResponse
    {
        $data = EmpEduMaster::where('Is_Active', 1)
            ->orderBy('EEd_Name')
            ->get();

        return response()->json([
            'success' => true,
            'data' => $data
        ]);
    }
}
