<?php

namespace App\Http\Controllers;

use App\Models\EmpLangMaster;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;

class EmpLangMasterController extends Controller
{
    /**
     * List with Search & Pagination
     */
    public function index(Request $request): JsonResponse
    {
        try {

            $perPage = $request->get('per_page', 10);
            $search = $request->get('search', '');

            $query = EmpLangMaster::query();

            // Search
            if (!empty($search)) {
                $query->where(function ($q) use ($search) {
                    $q->where('ELg_Code', 'like', "%{$search}%")
                        ->orWhere('Elg_Name', 'like', "%{$search}%");
                });
            }

            $data = $query->orderBy('ELg_Id', 'desc')
                ->paginate($perPage);

            return response()->json([
                'success' => true,
                'message' => 'Emp Language list fetched successfully',
                'data' => $data
            ]);
        } catch (\Exception $e) {

            return response()->json([
                'success' => false,
                'message' => $e->getMessage()
            ], 500);
        }
    }


    /**
     * Store
     */
    public function store(Request $request): JsonResponse
    {
        $validated = $request->validate([
            'ELg_Code' => 'nullable|string|max:10',
            'Elg_Name' => 'required|string|max:50',
            'Is_Active' => 'boolean'
        ]);

        $lang = EmpLangMaster::create($validated);

        return response()->json([
            'success' => true,
            'message' => 'Emp Language created successfully',
            'data' => $lang
        ]);
    }


    /**
     * Show Single
     */
    public function show($id): JsonResponse
    {
        $lang = EmpLangMaster::find($id);

        if (!$lang) {
            return response()->json([
                'success' => false,
                'message' => 'Record not found'
            ], 404);
        }

        return response()->json([
            'success' => true,
            'data' => $lang
        ]);
    }


    /**
     * Update
     */
    public function update(Request $request, $id): JsonResponse
    {
        $lang = EmpLangMaster::find($id);

        if (!$lang) {
            return response()->json([
                'success' => false,
                'message' => 'Record not found'
            ], 404);
        }

        $validated = $request->validate([
            'ELg_Code' => 'nullable|string|max:10',
            'Elg_Name' => 'required|string|max:50',
            'Is_Active' => 'boolean'
        ]);

        $lang->update($validated);

        return response()->json([
            'success' => true,
            'message' => 'Emp Language updated successfully',
            'data' => $lang
        ]);
    }


    /**
     * Delete (Soft Delete)
     */
    public function destroy($id): JsonResponse
    {
        $lang = EmpLangMaster::find($id);

        if (!$lang) {
            return response()->json([
                'success' => false,
                'message' => 'Record not found'
            ], 404);
        }

        $lang->delete();

        return response()->json([
            'success' => true,
            'message' => 'Emp Language deleted successfully'
        ]);
    }


    /**
     * All Active Records (Dropdown etc.)
     */
    public function all(): JsonResponse
    {
        $data = EmpLangMaster::where('Is_Active', 1)
            ->orderBy('Elg_Name')
            ->get();

        return response()->json([
            'success' => true,
            'data' => $data
        ]);
    }
}
