<?php

namespace App\Http\Controllers;

use App\Models\EmpQuotaMaster;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class EmpQuotaMasterController extends Controller
{
    /**
     * Display listing
     */
    public function index(Request $request)
    {
        try {

            $perPage = $request->get('per_page', 10);
            $search = $request->get('search', '');
            $withTrashed = $request->get('with_trashed', false);

            $query = EmpQuotaMaster::query();

            // Include soft deleted
            if ($withTrashed) {
                $query->withTrashed();
            }

            // Search
            if (!empty($search)) {
                $query->where(function ($q) use ($search) {
                    $q->where('Qut_Code', 'like', "%{$search}%")
                        ->orWhere('Qut_Name', 'like', "%{$search}%");
                });
            }

            $data = $query->orderBy('Qut_Id', 'desc')
                ->paginate($perPage);

            return response()->json([
                'status' => true,
                'message' => 'Quota list fetched successfully',
                'data' => $data
            ]);
        } catch (\Exception $e) {

            return response()->json([
                'status' => false,
                'message' => 'Something went wrong',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    public function all(Request $request)
    {
        try {

            $search = $request->get('search', '');
            $withTrashed = $request->get('with_trashed', false);

            $query = EmpQuotaMaster::query();

            if ($withTrashed) {
                $query->withTrashed();
            }

            if (!empty($search)) {
                $query->where(function ($q) use ($search) {
                    $q->where('Qut_Code', 'like', "%{$search}%")
                        ->orWhere('Qut_Name', 'like', "%{$search}%");
                });
            }

            $data = $query->orderBy('Qut_Id', 'desc')->get();

            return response()->json([
                'status' => true,
                'message' => 'All quota fetched successfully',
                'data' => $data
            ]);
        } catch (\Exception $e) {

            return response()->json([
                'status' => false,
                'message' => 'Something went wrong',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Store new record
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'Qut_Code' => 'nullable|string|max:20',
            'Qut_Name' => 'required|string|max:50',
            'Percentage' => 'nullable|numeric',
            'Is_Active' => 'nullable|boolean'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => false,
                'message' => $validator->errors()
            ], 422);
        }

        $data = EmpQuotaMaster::create([
            'Qut_Code' => $request->Qut_Code,
            'Qut_Name' => $request->Qut_Name,
            'Percentage' => $request->Percentage,
            'Is_Active' => $request->Is_Active ?? 1,
        ]);

        return response()->json([
            'status' => true,
            'message' => 'Quota created successfully',
            'data' => $data
        ]);
    }

    /**
     * Show single record
     */
    public function show($id)
    {
        $data = EmpQuotaMaster::find($id);

        if (!$data) {
            return response()->json([
                'status' => false,
                'message' => 'Quota not found'
            ], 404);
        }

        return response()->json([
            'status' => true,
            'data' => $data
        ]);
    }

    /**
     * Update record
     */
    public function update(Request $request, $id)
    {
        $data = EmpQuotaMaster::find($id);

        if (!$data) {
            return response()->json([
                'status' => false,
                'message' => 'Quota not found'
            ], 404);
        }

        $validator = Validator::make($request->all(), [
            'Qut_Code' => 'nullable|string|max:20',
            'Qut_Name' => 'required|string|max:50',
            'Percentage' => 'nullable|numeric',
            'Is_Active' => 'nullable|boolean'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => false,
                'message' => $validator->errors()
            ], 422);
        }

        $data->update([
            'Qut_Code' => $request->Qut_Code,
            'Qut_Name' => $request->Qut_Name,
            'Percentage' => $request->Percentage,
            'Is_Active' => $request->Is_Active ?? $data->Is_Active,
        ]);

        return response()->json([
            'status' => true,
            'message' => 'Quota updated successfully',
            'data' => $data
        ]);
    }

    /**
     * Soft Delete
     */
    public function destroy($id)
    {
        $data = EmpQuotaMaster::find($id);

        if (!$data) {
            return response()->json([
                'status' => false,
                'message' => 'Quota not found'
            ], 404);
        }

        $data->delete();

        return response()->json([
            'status' => true,
            'message' => 'Quota deleted successfully'
        ]);
    }
}
