<?php

namespace App\Http\Controllers;


use App\Models\EmpTypeMaster;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Throwable;

class EmpTypeMasterController extends Controller
{
    // 🔹 GET ALL
    public function index(Request $request)
    {
        try {
            $query = EmpTypeMaster::orderBy('ETy_Id', 'desc');

            $perPage = $request->get('per_page', 25);

            if ($perPage === 'all') {
                $data = $query->get();
            } else {
                $data = $query->paginate((int) $perPage);
            }

            return response()->json([
                'success' => true,
                'data' => $data
            ], 200);
        } catch (Throwable $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to fetch records',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    // LIST ALL (NO PAGINATION)
    public function all()
    {
        try {
            $data = EmpTypeMaster::orderBy('ETy_Id', 'desc')->get();

            return response()->json([
                'success' => true,
                'data' => $data
            ], 200);
        } catch (Throwable $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to fetch records',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    // 🔹 STORE
    public function store(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'ETy_Code' => 'nullable|string|max:20|unique:emp_type_master,ETy_Code',
                'ETy_Name' => 'required|string|max:50',
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'errors' => $validator->errors()
                ], 422);
            }

            $type = EmpTypeMaster::create([
                'ETy_Code' => $request->ETy_Code,
                'ETy_Name' => $request->ETy_Name,
                'Is_Active' => true
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Employee Type created successfully',
                'data' => $type
            ], 201);
        } catch (Throwable $e) {
            return response()->json([
                'success' => false,
                'message' => 'Create failed',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    // 🔹 SHOW
    public function show($id)
    {
        try {
            $type = EmpTypeMaster::find($id);

            if (!$type) {
                return response()->json([
                    'success' => false,
                    'message' => 'Record not found'
                ], 404);
            }

            return response()->json([
                'success' => true,
                'data' => $type
            ], 200);
        } catch (Throwable $e) {
            return response()->json([
                'success' => false,
                'error' => $e->getMessage()
            ], 500);
        }
    }

    // 🔹 UPDATE
    public function update(Request $request, $id)
    {
        try {
            $type = EmpTypeMaster::find($id);

            if (!$type) {
                return response()->json([
                    'success' => false,
                    'message' => 'Record not found'
                ], 404);
            }

            $validator = Validator::make($request->all(), [
                'ETy_Code' => 'sometimes|required|string|max:20|unique:emp_type_master,ETy_Code,' . $id . ',ETy_Id',
                'ETy_Name' => 'sometimes|required|string|max:50',
                'Is_Active' => 'sometimes|boolean'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'errors' => $validator->errors()
                ], 422);
            }

            $type->update($validator->validated());

            return response()->json([
                'success' => true,
                'message' => 'Employee Type updated successfully',
                'data' => $type
            ], 200);
        } catch (Throwable $e) {
            return response()->json([
                'success' => false,
                'message' => 'Update failed',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    // 🔹 DELETE
    public function destroy($id)
    {
        try {
            $type = EmpTypeMaster::find($id);

            if (!$type) {
                return response()->json([
                    'success' => false,
                    'message' => 'Record not found'
                ], 404);
            }

            $type->delete();

            return response()->json([
                'success' => true,
                'message' => 'Employee Type deleted successfully'
            ], 200);
        } catch (Throwable $e) {
            return response()->json([
                'success' => false,
                'message' => 'Delete failed',
                'error' => $e->getMessage()
            ], 500);
        }
    }
}
