<?php

namespace App\Http\Controllers;

use App\Models\EmployeeGradeMaster;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Throwable;

class EmployeeGradeMasterController extends Controller
{
    // 🔹 GET ALL
    public function index(Request $request)
    {
        try {
            $query = EmployeeGradeMaster::with('payBand');

            $perPage = $request->get('per_page', 25);

            if ($perPage === 'all') {
                $data = $query->get();
            } else {
                $data = $query->paginate((int) $perPage);
            }

            return response()->json([
                'success' => true,
                'data' => $data
            ], 200);
        } catch (Throwable $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to fetch records',
                'error' => $e->getMessage()
            ], 500);
        }
    }
    public function all()
    {
        try {
            $data = EmployeeGradeMaster::with('payBand')->get();

            return response()->json([
                'success' => true,
                'data' => $data
            ], 200);
        } catch (Throwable $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to fetch records',
                'error' => $e->getMessage()
            ], 500);
        }
    }


    // 🔹 STORE
    public function store(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'EGM_Code' => 'nullable|string|max:10|unique:emp_grade_master,EGM_Code',
                'EGM_Name' => 'required|string|max:50',
                'PB_ID'    => 'nullable|numeric',
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'errors' => $validator->errors()
                ], 422);
            }

            $grade = EmployeeGradeMaster::create([
                'EGM_Code' => $request->EGM_Code,
                'EGM_Name' => $request->EGM_Name,
                'PB_ID'    => $request->PB_ID,
                'Is_Active' => true
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Employee Grade created successfully',
                'data' => $grade
            ], 201);
        } catch (Throwable $e) {
            return response()->json([
                'success' => false,
                'message' => 'Create failed',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    // 🔹 SHOW
    public function show($id)
    {
        try {
            $grade = EmployeeGradeMaster::find($id);

            if (!$grade) {
                return response()->json([
                    'success' => false,
                    'message' => 'Record not found'
                ], 404);
            }

            return response()->json([
                'success' => true,
                'data' => $grade
            ], 200);
        } catch (Throwable $e) {
            return response()->json([
                'success' => false,
                'error' => $e->getMessage()
            ], 500);
        }
    }

    // 🔹 UPDATE
    public function update(Request $request, $id)
    {
        try {
            $grade = EmployeeGradeMaster::find($id);

            if (!$grade) {
                return response()->json([
                    'success' => false,
                    'message' => 'Record not found'
                ], 404);
            }

            $validator = Validator::make($request->all(), [
                'EGM_Code' => 'sometimes|required|string|max:10|unique:emp_grade_master,EGM_Code,' . $id . ',EGM_Id',
                'EGM_Name' => 'sometimes|required|string|max:50',
                'PB_ID'    => 'sometimes|nullable|numeric',
                'Is_Active' => 'sometimes|boolean'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'errors' => $validator->errors()
                ], 422);
            }

            $grade->update($validator->validated());

            return response()->json([
                'success' => true,
                'message' => 'Employee Grade updated successfully',
                'data' => $grade
            ], 200);
        } catch (Throwable $e) {
            return response()->json([
                'success' => false,
                'message' => 'Update failed',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    // 🔹 DELETE
    public function destroy($id)
    {
        try {
            $grade = EmployeeGradeMaster::find($id);

            if (!$grade) {
                return response()->json([
                    'success' => false,
                    'message' => 'Record not found'
                ], 404);
            }

            $grade->delete();

            return response()->json([
                'success' => true,
                'message' => 'Employee Grade deleted successfully'
            ], 200);
        } catch (Throwable $e) {
            return response()->json([
                'success' => false,
                'message' => 'Delete failed',
                'error' => $e->getMessage()
            ], 500);
        }
    }
}
