<?php

namespace App\Http\Controllers;

use App\Models\EmployeeStatusMaster;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Throwable;

class EmployeeStatusMasterController extends Controller
{
    // 🔹 INDEX
    public function index(Request $request)
    {
        try {
            $query = EmployeeStatusMaster::orderBy('Est_Id', 'desc');

            $perPage = $request->get('per_page', 25);

            if ($perPage === 'all') {
                $data = $query->get();
            } else {
                $data = $query->paginate((int) $perPage);
            }

            return response()->json([
                'success' => true,
                'message' => 'Employee statuses fetched successfully',
                'data' => $data
            ], 200);
        } catch (Throwable $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to fetch employee statuses',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    public function all()
    {
        try {
            $data = EmployeeStatusMaster::orderBy('Est_Id', 'desc')->get();

            return response()->json([
                'success' => true,
                'message' => 'Employee statuses fetched successfully',
                'data' => $data
            ], 200);
        } catch (Throwable $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to fetch employee statuses',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    // 🔹 STORE
    public function store(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'Est_Code' => 'nullable|string|max:20|unique:emp_status_master,Est_Code',
                'Est_Name' => 'required|string|max:50',
                'Is_Active' => 'boolean'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation error',
                    'errors'  => $validator->errors()
                ], 422);
            }

            $status = EmployeeStatusMaster::create([
                'Est_Code' => $request->Est_Code,
                'Est_Name' => $request->Est_Name,
                'Is_Active' => $request->Is_Active ?? 1
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Employee status created successfully',
                'data' => $status
            ], 201);
        } catch (Throwable $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to create employee status',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    // 🔹 SHOW
    public function show($id)
    {
        try {
            $status = EmployeeStatusMaster::find($id);

            if (!$status) {
                return response()->json([
                    'success' => false,
                    'message' => 'Employee status not found'
                ], 404);
            }

            return response()->json([
                'success' => true,
                'data' => $status
            ], 200);
        } catch (Throwable $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to fetch employee status',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    // 🔹 UPDATE
    public function update(Request $request, $id)
    {
        try {
            $status = EmployeeStatusMaster::find($id);

            if (!$status) {
                return response()->json([
                    'success' => false,
                    'message' => 'Employee status not found'
                ], 404);
            }

            $validator = Validator::make($request->all(), [
                'Est_Code' => 'sometimes|required|string|max:20|unique:emp_status_master,Est_Code,' . $id . ',Est_Id',
                'Est_Name' => 'sometimes|required|string|max:50',
                'Is_Active' => 'sometimes|boolean'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation error',
                    'errors'  => $validator->errors()
                ], 422);
            }

            $status->update($validator->validated());

            return response()->json([
                'success' => true,
                'message' => 'Employee status updated successfully',
                'data' => $status
            ], 200);
        } catch (Throwable $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to update employee status',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    // 🔹 DELETE
    public function destroy($id)
    {
        try {
            $status = EmployeeStatusMaster::find($id);

            if (!$status) {
                return response()->json([
                    'success' => false,
                    'message' => 'Employee status not found'
                ], 404);
            }

            $status->delete();

            return response()->json([
                'success' => true,
                'message' => 'Employee status deleted successfully'
            ], 200);
        } catch (Throwable $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to delete employee status',
                'error' => $e->getMessage()
            ], 500);
        }
    }
}
