<?php

namespace App\Http\Controllers;

use App\Models\FundsTypeMaster;
use App\Models\GrantTypeMaster;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class FundsTypeMasterController extends Controller
{
    public function index()
    {
        $fundsTypes = FundsTypeMaster::with('grantType')
            ->where('Is_Active', true)
            ->get();

        return response()->json([
            'success' => true,
            'data' => $fundsTypes
        ]);
    }

    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'FT_Code'     => 'nullable|string|max:20|unique:funds_type_master,FT_Code',
            'FT_Name'     => 'required|string|max:100',
            'FT_Type'     => 'nullable|string|max:50',
            'FT_Source'   => 'nullable|string|max:50',
            'GT_Id'       => 'nullable|exists:grant_type_master,GT_Id',
            'FT_Category' => 'nullable|string|max:20',
            'Is_Active'   => 'boolean'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors()
            ], 422);
        }

        // Validate GT_Id exists if provided
        if ($request->has('GT_Id') && $request->GT_Id) {
            $grantType = GrantTypeMaster::find($request->GT_Id);
            if (!$grantType) {
                return response()->json([
                    'success' => false,
                    'message' => 'Grant type not found'
                ], 404);
            }
        }

        $fundsType = FundsTypeMaster::create([
            'FT_Code'     => $request->FT_Code,
            'FT_Name'     => $request->FT_Name,
            'FT_Type'     => $request->FT_Type,
            'FT_Source'   => $request->FT_Source,
            'GT_Id'       => $request->GT_Id,
            'FT_Category' => $request->FT_Category,
            'Is_Active'   => $request->has('Is_Active') ? $request->Is_Active : true
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Funds type created successfully',
            'data' => $fundsType->load('grantType')
        ], 201);
    }

    public function show($id)
    {
        $fundsType = FundsTypeMaster::with('grantType')->find($id);

        if (!$fundsType) {
            return response()->json([
                'success' => false,
                'message' => 'Funds type not found'
            ], 404);
        }

        return response()->json([
            'success' => true,
            'data' => $fundsType
        ]);
    }

    public function update(Request $request, $id)
    {
        $fundsType = FundsTypeMaster::find($id);

        if (!$fundsType) {
            return response()->json([
                'success' => false,
                'message' => 'Funds type not found'
            ], 404);
        }

        $validator = Validator::make($request->all(), [
            'FT_Code'     => 'nullable|string|max:20|unique:funds_type_master,FT_Code,' . $id . ',FT_id',
            'FT_Name'     => 'sometimes|required|string|max:100',
            'FT_Type'     => 'nullable|string|max:50',
            'FT_Source'   => 'nullable|string|max:50',
            'GT_Id'       => 'nullable|exists:grant_type_master,GT_Id',
            'FT_Category' => 'nullable|string|max:20',
            'Is_Active'   => 'boolean'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors()
            ], 422);
        }

        // Validate GT_Id exists if provided
        if ($request->has('GT_Id') && $request->GT_Id) {
            $grantType = GrantTypeMaster::find($request->GT_Id);
            if (!$grantType) {
                return response()->json([
                    'success' => false,
                    'message' => 'Grant type not found'
                ], 404);
            }
        }

        $fundsType->update([
            'FT_Code'     => $request->FT_Code ?? $fundsType->FT_Code,
            'FT_Name'     => $request->FT_Name ?? $fundsType->FT_Name,
            'FT_Type'     => $request->FT_Type ?? $fundsType->FT_Type,
            'FT_Source'   => $request->FT_Source ?? $fundsType->FT_Source,
            'GT_Id'       => $request->GT_Id ?? $fundsType->GT_Id,
            'FT_Category' => $request->FT_Category ?? $fundsType->FT_Category,
            'Is_Active'   => $request->has('Is_Active') ? $request->Is_Active : $fundsType->Is_Active
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Funds type updated successfully',
            'data' => $fundsType->load('grantType')
        ]);
    }

    public function destroy($id)
    {
        $fundsType = FundsTypeMaster::find($id);

        if (!$fundsType) {
            return response()->json([
                'success' => false,
                'message' => 'Funds type not found'
            ], 404);
        }

        $fundsType->update(['Is_Active' => false]);

        return response()->json([
            'success' => true,
            'message' => 'Funds type deactivated successfully'
        ]);
    }

    public function restore($id)
    {
        $fundsType = FundsTypeMaster::find($id);

        if (!$fundsType) {
            return response()->json([
                'success' => false,
                'message' => 'Funds type not found'
            ], 404);
        }

        $fundsType->update(['Is_Active' => true]);

        return response()->json([
            'success' => true,
            'message' => 'Funds type activated successfully',
            'data' => $fundsType->load('grantType')
        ]);
    }

    /**
     * Get funds types by grant type
     */
    public function getByGrantType($grantTypeId)
    {
        $grantType = GrantTypeMaster::find($grantTypeId);

        if (!$grantType) {
            return response()->json([
                'success' => false,
                'message' => 'Grant type not found'
            ], 404);
        }

        $fundsTypes = FundsTypeMaster::with('grantType')
            ->where('GT_Id', $grantTypeId)
            ->where('Is_Active', true)
            ->get();

        return response()->json([
            'success' => true,
            'data' => $fundsTypes,
            'grant_type' => $grantType
        ]);
    }

    /**
     * Get all funds types (including inactive)
     */
    public function getAll()
    {
        $fundsTypes = FundsTypeMaster::with('grantType')->get();

        return response()->json([
            'success' => true,
            'data' => $fundsTypes
        ]);
    }
}
