<?php

namespace App\Http\Controllers;

use App\Http\Controllers\Controller;
use App\Models\GrantTypeMaster;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class GrantTypeMasterController extends Controller
{
  public function index(Request $request)
{
    try {
        $perPage = $request->get('per_page', 25);
        $search = $request->get('search', '');
        $withTrashed = $request->get('with_trashed', false);

        $query = GrantTypeMaster::with('grantAuthority')
            ->where('Is_Active', true)
            ->orderBy('GT_Id', 'desc'); // keep ordering

        // Include soft deleted if requested
        if ($withTrashed) {
            $query->withTrashed();
        }

        // Search filter
        if ($search) {
            $query->where(function ($q) use ($search) {
                $q->where('GT_Name', 'like', "%{$search}%")
                  ->orWhere('GT_Id', 'like', "%{$search}%");
            });
        }

        // Pagination or all
        if ($perPage === 'all') {
            $data = $query->get();
        } else {
            $data = $query->paginate((int) $perPage);
        }

        return response()->json([
            'success' => true,
            'message' => 'Grant types fetched successfully',
            'data' => $data
        ], 200);

    } catch (\Exception $e) {
        return response()->json([
            'success' => false,
            'message' => 'Something went wrong',
            'error' => $e->getMessage()
        ], 500);
    }
}
public function all()
{
    try {
        return response()->json([
            'status' => true,
            'data' => GrantTypeMaster::with('grantAuthority')
                        ->where('Is_Active', true)
                        ->latest()
                        ->get()
        ]);
    } catch (\Exception $e) {
        return response()->json([
            'status' => false,
            'message' => $e->getMessage()
        ], 500);
    }
}


    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'GT_Code' => 'nullable|string|max:25|unique:grant_type_master,GT_Code',
            'GT_Name' => 'required|string|max:100',
            'GT_Nature' => 'nullable|string|max:25|in:Revenue,Capital',
            'GT_Condition' => 'nullable|string|in:Y,N',
            'GT_Utl_From_Date' => 'nullable|date|before_or_equal:GT_Utl_To_Date',
            'GT_Utl_To_Date' => 'nullable|date|after_or_equal:GT_Utl_From_Date',
            'GT_Act_Mode' => 'nullable|string|max:50|in:Income,Liability,Differed',
            'GT_Status' => 'nullable|string|max:25',
            'GT_Au_Id' => 'nullable|exists:grant_autho_master,GA_Id',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors()
            ], 422);
        }

        $grantType = GrantTypeMaster::create(array_merge(
            $request->all(),
            ['Is_Active' => true]
        ));

        $grantType->load('grantAuthority');

        return response()->json([
            'success' => true,
            'message' => 'Grant type created successfully',
            'data' => $grantType
        ], 201);
    }

    public function show($id)
    {
        $grantType = GrantTypeMaster::with('grantAuthority')->find($id);

        if (!$grantType) {
            return response()->json([
                'success' => false,
                'message' => 'Grant type not found'
            ], 404);
        }

        return response()->json([
            'success' => true,
            'data' => $grantType
        ]);
    }

    public function update(Request $request, $id)
    {
        $grantType = GrantTypeMaster::find($id);

        if (!$grantType) {
            return response()->json([
                'success' => false,
                'message' => 'Grant type not found'
            ], 404);
        }

        // Manual validation for unique GT_Code
        if ($request->has('GT_Code') && $request->GT_Code !== $grantType->GT_Code) {
            $exists = GrantTypeMaster::where('GT_Code', $request->GT_Code)
                ->where('GT_Id', '!=', $id)
                ->exists();

            if ($exists) {
                return response()->json([
                    'success' => false,
                    'errors' => ['GT_Code' => ['The grant code has already been taken.']]
                ], 422);
            }
        }

        $validator = Validator::make($request->all(), [
            'GT_Code' => 'required|string|max:25',
            'GT_Name' => 'required|string|max:100',
            'GT_Nature' => 'nullable|string|max:25|in:Revenue,Capital',
            'GT_Condition' => 'nullable|string|in:Y,N',
            'GT_Utl_From_Date' => 'nullable|date|before_or_equal:GT_Utl_To_Date',
            'GT_Utl_To_Date' => 'nullable|date|after_or_equal:GT_Utl_From_Date',
            'GT_Act_Mode' => 'nullable|string|max:50|in:Income,Liability,Differed',
            'GT_Status' => 'nullable|string|max:25',
            'GT_Au_Id' => 'nullable|exists:grant_autho_master,GA_Id',
            'Is_Active' => 'nullable|boolean'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors()
            ], 422);
        }

        $grantType->update($request->all());

        return response()->json([
            'success' => true,
            'message' => 'Grant type updated successfully',
            'data' => $grantType
        ]);
    }

    public function destroy($id)
    {
        $grantType = GrantTypeMaster::find($id);

        if (!$grantType) {
            return response()->json([
                'success' => false,
                'message' => 'Grant type not found'
            ], 404);
        }

        $grantType->update(['Is_Active' => false]);

        return response()->json([
            'success' => true,
            'message' => 'Grant type deactivated successfully'
        ]);
    }

    public function restore($id)
    {
        $grantType = GrantTypeMaster::find($id);

        if (!$grantType) {
            return response()->json([
                'success' => false,
                'message' => 'Grant type not found'
            ], 404);
        }

        $grantType->update(['Is_Active' => true]);

        return response()->json([
            'success' => true,
            'message' => 'Grant type activated successfully',
            'data' => $grantType
        ]);
    }
}
