<?php

namespace App\Http\Controllers;

use App\Models\HolidayDsgMap;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Database\Eloquent\ModelNotFoundException;

class HolidayDsgMapController extends Controller
{
  public function index(Request $request)
{
    try {
        $perPage = $request->get('per_page', 15);
        $maps = HolidayDsgMap::with(['holidayMaster', 'designationMaster'])->paginate($perPage);

        // Debug: Check if designation exists
        foreach ($maps as $map) {
            \Log::info('Map ID: ' . $map->HLDS_Id . ', Dsg_Id: ' . $map->Dsg_Id);
            if ($map->Dsg_Id) {
                $designation = \App\Models\DesignationMaster::find($map->Dsg_Id);
                \Log::info('Designation found: ' . ($designation ? 'Yes' : 'No'));
            }
        }

        return response()->json([
            'success' => true,
            'data' => $maps,
            'message' => 'Holiday designation maps retrieved successfully'
        ], 200);
    } catch (\Exception $e) {
        return response()->json([
            'success' => false,
            'message' => 'Error retrieving maps',
            'error' => $e->getMessage()
        ], 500);
    }
}

public function all()
{
    try {
        $maps = HolidayDsgMap::with(['holidayMaster', 'designationMaster'])->get();

        return response()->json([
            'success' => true,
            'data' => $maps,
            'message' => 'All holiday designation maps retrieved successfully'
        ], 200);
    } catch (\Exception $e) {
        return response()->json([
            'success' => false,
            'message' => 'Error retrieving maps',
            'error' => $e->getMessage()
        ], 500);
    }
}

public function show($id)
{
    try {
        $map = HolidayDsgMap::with(['holidayMaster', 'designationMaster'])->findOrFail($id);

        return response()->json([
            'success' => true,
            'data' => $map,
            'message' => 'Holiday designation map retrieved successfully'
        ], 200);
    } catch (ModelNotFoundException $e) {
        return response()->json([
            'success' => false,
            'message' => 'Map not found'
        ], 404);
    } catch (\Exception $e) {
        return response()->json([
            'success' => false,
            'message' => 'Error retrieving map',
            'error' => $e->getMessage()
        ], 500);
    }
}



    public function store(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'HL_Id' => 'nullable|integer|exists:holiday_master,HL_Id',
                'Dsg_Id' => 'nullable|integer|exists:designation_master,Dsg_Id',
                'HDS_Status' => 'nullable|boolean'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation errors',
                    'errors' => $validator->errors()
                ], 422);
            }

            $map = HolidayDsgMap::create($request->all());

            return response()->json([
                'success' => true,
                'data' => $map,
                'message' => 'Holiday designation map created successfully'
            ], 201);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error creating map',
                'error' => $e->getMessage()
            ], 500);
        }
    }

   

    public function update(Request $request, $id)
    {
        try {
            $validator = Validator::make($request->all(), [
                'HL_Id' => 'nullable|integer|exists:holiday_master,HL_Id',
                'Dsg_Id' => 'nullable|integer|exists:designation_master,Dsg_Id',
                'HDS_Status' => 'nullable|boolean'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation errors',
                    'errors' => $validator->errors()
                ], 422);
            }

            $map = HolidayDsgMap::findOrFail($id);
            $map->update($request->all());

            return response()->json([
                'success' => true,
                'data' => $map,
                'message' => 'Holiday designation map updated successfully'
            ], 200);
        } catch (ModelNotFoundException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Map not found'
            ], 404);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error updating map',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    public function destroy($id)
    {
        try {
            $map = HolidayDsgMap::findOrFail($id);
            $map->delete();

            return response()->json([
                'success' => true,
                'message' => 'Holiday designation map deleted successfully'
            ], 200);
        } catch (ModelNotFoundException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Map not found'
            ], 404);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error deleting map',
                'error' => $e->getMessage()
            ], 500);
        }
    }
}
