<?php

namespace App\Http\Controllers;

use App\Models\HolidayEmpTyMap;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Database\Eloquent\ModelNotFoundException;

class HolidayEmpTyMapController extends Controller
{
   public function index(Request $request)
{
    try {
        $perPage = $request->get('per_page', 15);
        $maps = HolidayEmpTyMap::with(['holidayMaster', 'empTypeMaster'])->paginate($perPage);

        return response()->json([
            'success' => true,
            'data' => $maps,
            'message' => 'Holiday employee type maps retrieved successfully'
        ], 200);
    } catch (\Exception $e) {
        return response()->json([
            'success' => false,
            'message' => 'Error retrieving maps',
            'error' => $e->getMessage()
        ], 500);
    }
}

public function all()
{
    try {
        $maps = HolidayEmpTyMap::with(['holidayMaster', 'empTypeMaster'])->get();

        return response()->json([
            'success' => true,
            'data' => $maps,
            'message' => 'All holiday employee type maps retrieved successfully'
        ], 200);
    } catch (\Exception $e) {
        return response()->json([
            'success' => false,
            'message' => 'Error retrieving maps',
            'error' => $e->getMessage()
        ], 500);
    }
}

public function show($id)
{
    try {
        $map = HolidayEmpTyMap::with(['holidayMaster', 'empTypeMaster'])->findOrFail($id);

        return response()->json([
            'success' => true,
            'data' => $map,
            'message' => 'Holiday employee type map retrieved successfully'
        ], 200);
    } catch (ModelNotFoundException $e) {
        return response()->json([
            'success' => false,
            'message' => 'Map not found'
        ], 404);
    } catch (\Exception $e) {
        return response()->json([
            'success' => false,
            'message' => 'Error retrieving map',
            'error' => $e->getMessage()
        ], 500);
    }
}

    public function store(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'HL_Id' => 'required|integer|exists:holiday_master,HL_Id',
                'ETy_Id' => 'required|integer|exists:emp_type_master,ETy_Id',
                'HET_Status' => 'nullable|boolean'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation errors',
                    'errors' => $validator->errors()
                ], 422);
            }

            $map = HolidayEmpTyMap::create($request->all());

            return response()->json([
                'success' => true,
                'data' => $map,
                'message' => 'Holiday employee type map created successfully'
            ], 201);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error creating map',
                'error' => $e->getMessage()
            ], 500);
        }
    }

   
    public function update(Request $request, $id)
    {
        try {
            $validator = Validator::make($request->all(), [
                'HL_Id' => 'required|integer|exists:holiday_master,HL_Id',
                'ETy_Id' => 'required|integer|exists:emp_type_master,ETy_Id',
                'HET_Status' => 'nullable|boolean'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation errors',
                    'errors' => $validator->errors()
                ], 422);
            }

            $map = HolidayEmpTyMap::findOrFail($id);
            $map->update($request->all());

            return response()->json([
                'success' => true,
                'data' => $map,
                'message' => 'Holiday employee type map updated successfully'
            ], 200);
        } catch (ModelNotFoundException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Map not found'
            ], 404);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error updating map',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    public function destroy($id)
    {
        try {
            $map = HolidayEmpTyMap::findOrFail($id);
            $map->delete();

            return response()->json([
                'success' => true,
                'message' => 'Holiday employee type map deleted successfully'
            ], 200);
        } catch (ModelNotFoundException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Map not found'
            ], 404);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error deleting map',
                'error' => $e->getMessage()
            ], 500);
        }
    }
}
