<?php
// app/Http/Controllers/HolidayMasterController.php

namespace App\Http\Controllers;

use App\Models\HolidayMaster;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Database\Eloquent\ModelNotFoundException;

class HolidayMasterController extends Controller
{
    /**
     * Display a listing of the resource with pagination.
     */
    public function index(Request $request)
    {
        try {
            $perPage = $request->get('per_page', 15);
            $holidays = HolidayMaster::paginate($perPage);

            return response()->json([
                'success' => true,
                'data' => $holidays,
                'message' => 'Holidays retrieved successfully'
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error retrieving holidays',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get all holidays without pagination.
     */
    public function all()
    {
        try {
            $holidays = HolidayMaster::all();

            return response()->json([
                'success' => true,
                'data' => $holidays,
                'message' => 'All holidays retrieved successfully'
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error retrieving holidays',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Store a newly created holiday in storage.
     */
    public function store(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'HL_Code' => 'nullable|string|max:255',
                'HL_Name' => 'nullable|string|max:255',
                'CI_Year' => 'nullable|string|max:255',
                'HL_Date' => 'nullable|string|max:255',
                'HL_Type' => 'nullable|string|max:255',
                'HL_Cate' => 'nullable|string|max:255',
                'HL_ISPaid' => 'nullable|string|max:255',
                'HL_ISOptional' => 'nullable|string|max:255',
                'HL_ISCmpnst' => 'nullable|string|max:255',
                'HL_AttnImpact' => 'nullable|string|max:255',
                'HL_PayImpact' => 'nullable|string|max:255',
                'HL_OTAllowed' => 'nullable|string|max:255',
                'HL_Status' => 'nullable|boolean',
                'Remarks' => 'nullable|string|max:255'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation errors',
                    'errors' => $validator->errors()
                ], 422);
            }

            $holiday = HolidayMaster::create($request->all());

            return response()->json([
                'success' => true,
                'data' => $holiday,
                'message' => 'Holiday created successfully'
            ], 201);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error creating holiday',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Display the specified holiday.
     */
    public function show($id)
    {
        try {
            $holiday = HolidayMaster::findOrFail($id);

            return response()->json([
                'success' => true,
                'data' => $holiday,
                'message' => 'Holiday retrieved successfully'
            ], 200);
        } catch (ModelNotFoundException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Holiday not found'
            ], 404);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error retrieving holiday',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Update the specified holiday in storage.
     */
    public function update(Request $request, $id)
    {
        try {
            $validator = Validator::make($request->all(), [
                'HL_Code' => 'nullable|string|max:255',
                'HL_Name' => 'nullable|string|max:255',
                'CI_Year' => 'nullable|string|max:255',
                'HL_Date' => 'nullable|string|max:255',
                'HL_Type' => 'nullable|string|max:255',
                'HL_Cate' => 'nullable|string|max:255',
                'HL_ISPaid' => 'nullable|string|max:255',
                'HL_ISOptional' => 'nullable|string|max:255',
                'HL_ISCmpnst' => 'nullable|string|max:255',
                'HL_AttnImpact' => 'nullable|string|max:255',
                'HL_PayImpact' => 'nullable|string|max:255',
                'HL_OTAllowed' => 'nullable|string|max:255',
                'HL_Status' => 'nullable|boolean',
                'Remarks' => 'nullable|string|max:255'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation errors',
                    'errors' => $validator->errors()
                ], 422);
            }

            $holiday = HolidayMaster::findOrFail($id);
            $holiday->update($request->all());

            return response()->json([
                'success' => true,
                'data' => $holiday,
                'message' => 'Holiday updated successfully'
            ], 200);
        } catch (ModelNotFoundException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Holiday not found'
            ], 404);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error updating holiday',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Remove the specified holiday from storage.
     */
    public function destroy($id)
    {
        try {
            $holiday = HolidayMaster::findOrFail($id);
            $holiday->delete();

            return response()->json([
                'success' => true,
                'message' => 'Holiday deleted successfully'
            ], 200);
        } catch (ModelNotFoundException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Holiday not found'
            ], 404);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error deleting holiday',
                'error' => $e->getMessage()
            ], 500);
        }
    }
}
