<?php

namespace App\Http\Controllers;

use App\Models\HolidayShiftMap;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Database\Eloquent\ModelNotFoundException;

class HolidayShiftMapController extends Controller
{
   public function index(Request $request)
{
    try {
        $perPage = $request->get('per_page', 15);
        $maps = HolidayShiftMap::with(['holidayMaster', 'shiftMaster'])->paginate($perPage);

        return response()->json([
            'success' => true,
            'data' => $maps,
            'message' => 'Holiday shift maps retrieved successfully'
        ], 200);
    } catch (\Exception $e) {
        return response()->json([
            'success' => false,
            'message' => 'Error retrieving maps',
            'error' => $e->getMessage()
        ], 500);
    }
}

public function all()
{
    try {
        $maps = HolidayShiftMap::with(['holidayMaster', 'shiftMaster'])->get();

        return response()->json([
            'success' => true,
            'data' => $maps,
            'message' => 'All holiday shift maps retrieved successfully'
        ], 200);
    } catch (\Exception $e) {
        return response()->json([
            'success' => false,
            'message' => 'Error retrieving maps',
            'error' => $e->getMessage()
        ], 500);
    }
}

public function show($id)
{
    try {
        $map = HolidayShiftMap::with(['holidayMaster', 'shiftMaster'])->findOrFail($id);

        return response()->json([
            'success' => true,
            'data' => $map,
            'message' => 'Holiday shift map retrieved successfully'
        ], 200);
    } catch (ModelNotFoundException $e) {
        return response()->json([
            'success' => false,
            'message' => 'Map not found'
        ], 404);
    } catch (\Exception $e) {
        return response()->json([
            'success' => false,
            'message' => 'Error retrieving map',
            'error' => $e->getMessage()
        ], 500);
    }
}

    public function store(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'HL_Id' => 'nullable|integer|exists:holiday_master,HL_Id',
                'Sf_Id' => 'nullable|integer|exists:shift_master,Sf_Id',
                'HDS_Status' => 'nullable|boolean'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation errors',
                    'errors' => $validator->errors()
                ], 422);
            }

            $map = HolidayShiftMap::create($request->all());

            return response()->json([
                'success' => true,
                'data' => $map,
                'message' => 'Holiday shift map created successfully'
            ], 201);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error creating map',
                'error' => $e->getMessage()
            ], 500);
        }
    }

   

    public function update(Request $request, $id)
    {
        try {
            $validator = Validator::make($request->all(), [
                'HL_Id' => 'nullable|integer|exists:holiday_master,HL_Id',
                'Sf_Id' => 'nullable|integer|exists:shift_master,Sf_Id',
                'HDS_Status' => 'nullable|boolean'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation errors',
                    'errors' => $validator->errors()
                ], 422);
            }

            $map = HolidayShiftMap::findOrFail($id);
            $map->update($request->all());

            return response()->json([
                'success' => true,
                'data' => $map,
                'message' => 'Holiday shift map updated successfully'
            ], 200);
        } catch (ModelNotFoundException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Map not found'
            ], 404);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error updating map',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    public function destroy($id)
    {
        try {
            $map = HolidayShiftMap::findOrFail($id);
            $map->delete();

            return response()->json([
                'success' => true,
                'message' => 'Holiday shift map deleted successfully'
            ], 200);
        } catch (ModelNotFoundException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Map not found'
            ], 404);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error deleting map',
                'error' => $e->getMessage()
            ], 500);
        }
    }
}
