<?php

namespace App\Http\Controllers;

use App\Models\IncExamMaster;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Database\Eloquent\ModelNotFoundException;

class IncExamMasterController extends Controller
{
    public function index(Request $request)
    {
        try {
            $limit = $request->get('limit', 10);
            $search = $request->get('search', '');

            $query = IncExamMaster::query();

            if (!empty($search)) {
                $query->where(function ($q) use ($search) {
                    $q->where('EEd_Name', 'LIKE', "%{$search}%")
                        ->orWhere('EEd_Code', 'LIKE', "%{$search}%");
                });
            }

            $exams = $query->orderBy('EEd_Id', 'desc')->paginate($limit);

            return response()->json([
                'success' => true,
                'message' => 'Exams retrieved successfully',
                'data' => $exams->items(),
                'pagination' => [
                    'current_page' => $exams->currentPage(),
                    'last_page' => $exams->lastPage(),
                    'per_page' => $exams->perPage(),
                    'total' => $exams->total(),
                    'next_page_url' => $exams->nextPageUrl(),
                    'prev_page_url' => $exams->previousPageUrl()
                ]
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to retrieve exams',
                'error' => $e->getMessage()
            ], 500);
        }
    }


    public function getAllExams()
    {
        try {
            $exams = IncExamMaster::where('Is_Active', 1)
                                 ->orderBy('EEd_Name', 'asc')
                                 ->get(['EEd_Id', 'EEd_Code', 'EEd_Name', 'Is_Active']);

            return response()->json([
                'success' => true,
                'message' => 'All exams retrieved successfully',
                'total' => $exams->count(),
                'data' => $exams
            ], 200);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to retrieve exams',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    public function store(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'EEd_Code' => 'nullable|string|max:20',
                'EEd_Name' => 'required|string|max:100',
                'Is_Active' => 'sometimes|boolean',
                'Is_Mandatory' => 'sometimes|in:Y,N',
                'Is_Technical' => 'sometimes|in:Y,N',
                'Is_Professional' => 'sometimes|in:Y,N',
                'Is_Promotional' => 'sometimes|in:Y,N',
                'Rec_Valid' => 'sometimes|in:Y,N'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'errors' => $validator->errors()
                ], 422);
            }

            $data = [
                'EEd_Code' => $request->EEd_Code,
                'EEd_Name' => $request->EEd_Name,
                'Is_Active' => $request->Is_Active ?? 1,
                'Is_Mandatory' => $request->Is_Mandatory ?? 'N',
                'Is_Technical' => $request->Is_Technical ?? 'N',
                'Is_Professional' => $request->Is_Professional ?? 'N',
                'Is_Promotional' => $request->Is_Promotional ?? 'N',
                'Rec_Valid' => $request->Rec_Valid ?? 'N'
            ];

            $exam = IncExamMaster::create($data);

            return response()->json([
                'success' => true,
                'message' => 'Exam created successfully',
                'data' => $exam
            ], 201);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to create exam',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    public function show($id)
    {
        try {
            $exam = IncExamMaster::findOrFail($id);
            return response()->json([
                'success' => true,
                'data' => $exam
            ], 200);
        } catch (ModelNotFoundException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Exam not found'
            ], 404);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to retrieve exam',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    public function update(Request $request, $id)
    {
        try {
            $exam = IncExamMaster::findOrFail($id);

            $validator = Validator::make($request->all(), [
                'EEd_Code' => 'nullable|string|max:20',
                'EEd_Name' => 'sometimes|required|string|max:100',
                'Is_Active' => 'sometimes|boolean',
                'Is_Mandatory' => 'sometimes|in:Y,N',
                'Is_Technical' => 'sometimes|in:Y,N',
                'Is_Professional' => 'sometimes|in:Y,N',
                'Is_Promotional' => 'sometimes|in:Y,N',
                'Rec_Valid' => 'sometimes|in:Y,N'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'errors' => $validator->errors()
                ], 422);
            }

            $data = [
                'EEd_Code' => $request->has('EEd_Code') ? $request->EEd_Code : $exam->EEd_Code,
                'EEd_Name' => $request->has('EEd_Name') ? $request->EEd_Name : $exam->EEd_Name,
                'Is_Active' => $request->has('Is_Active') ? $request->Is_Active : $exam->Is_Active,
                'Is_Mandatory' => $request->has('Is_Mandatory') ? $request->Is_Mandatory : $exam->Is_Mandatory,
                'Is_Technical' => $request->has('Is_Technical') ? $request->Is_Technical : $exam->Is_Technical,
                'Is_Professional' => $request->has('Is_Professional') ? $request->Is_Professional : $exam->Is_Professional,
                'Is_Promotional' => $request->has('Is_Promotional') ? $request->Is_Promotional : $exam->Is_Promotional,
                'Rec_Valid' => $request->has('Rec_Valid') ? $request->Rec_Valid : $exam->Rec_Valid
            ];

            $exam->update($data);

            return response()->json([
                'success' => true,
                'message' => 'Exam updated successfully',
                'data' => $exam
            ], 200);
        } catch (ModelNotFoundException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Exam not found'
            ], 404);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to update exam',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    public function destroy($id)
    {
        try {
            $exam = IncExamMaster::findOrFail($id);
            $exam->delete();

            return response()->json([
                'success' => true,
                'message' => 'Exam deleted successfully'
            ], 200);
        } catch (ModelNotFoundException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Exam not found'
            ], 404);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to delete exam',
                'error' => $e->getMessage()
            ], 500);
        }
    }
}
