<?php

namespace App\Http\Controllers;

use App\Models\LeaveTypeMaster;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Database\Eloquent\ModelNotFoundException;

class LeaveTypeMasterController extends Controller
{
    public function index(Request $request)
    {
        try {
            $perPage = $request->get('per_page', 15);
            $leaveTypes = LeaveTypeMaster::paginate($perPage);

            return response()->json([
                'success' => true,
                'data' => $leaveTypes,
                'message' => 'Leave types retrieved successfully'
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error retrieving leave types',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    public function all()
    {
        try {
            $leaveTypes = LeaveTypeMaster::all();

            return response()->json([
                'success' => true,
                'data' => $leaveTypes,
                'message' => 'All leave types retrieved successfully'
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error retrieving leave types',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    public function store(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'LT_code' => 'nullable|string|max:20',
                'LT_name' => 'required|string|max:100',
                'Max_days' => 'nullable|integer',
                'Is_Paid' => 'nullable|boolean',
                'Is_Active' => 'nullable|boolean'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation errors',
                    'errors' => $validator->errors()
                ], 422);
            }

            $leaveType = LeaveTypeMaster::create($request->all());

            return response()->json([
                'success' => true,
                'data' => $leaveType,
                'message' => 'Leave type created successfully'
            ], 201);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error creating leave type',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    public function show($id)
    {
        try {
            $leaveType = LeaveTypeMaster::findOrFail($id);

            return response()->json([
                'success' => true,
                'data' => $leaveType,
                'message' => 'Leave type retrieved successfully'
            ], 200);
        } catch (ModelNotFoundException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Leave type not found'
            ], 404);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error retrieving leave type',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    public function update(Request $request, $id)
    {
        try {
            $validator = Validator::make($request->all(), [
                'LT_code' => 'nullable|string|max:20',
                'LT_name' => 'required|string|max:100',
                'Max_days' => 'nullable|integer',
                'Is_Paid' => 'nullable|boolean',
                'Is_Active' => 'nullable|boolean'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation errors',
                    'errors' => $validator->errors()
                ], 422);
            }

            $leaveType = LeaveTypeMaster::findOrFail($id);
            $leaveType->update($request->all());

            return response()->json([
                'success' => true,
                'data' => $leaveType,
                'message' => 'Leave type updated successfully'
            ], 200);
        } catch (ModelNotFoundException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Leave type not found'
            ], 404);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error updating leave type',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    public function destroy($id)
    {
        try {
            $leaveType = LeaveTypeMaster::findOrFail($id);
            $leaveType->delete();

            return response()->json([
                'success' => true,
                'message' => 'Leave type deleted successfully'
            ], 200);
        } catch (ModelNotFoundException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Leave type not found'
            ], 404);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error deleting leave type',
                'error' => $e->getMessage()
            ], 500);
        }
    }
}
