<?php

namespace App\Http\Controllers;

use App\Models\OrgMaster;
use Illuminate\Http\Request;
use Illuminate\Http\Response;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;

class OrgMasterController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        try {
            $orgMasters = OrgMaster::all();

            return response()->json([
                'success' => true,
                'message' => 'Organisations retrieved successfully',
                'data' => $orgMasters
            ], Response::HTTP_OK);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to retrieve organisations',
                'error' => $e->getMessage()
            ], Response::HTTP_INTERNAL_SERVER_ERROR);
        }
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), OrgMaster::createRules());

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation error',
                    'errors' => $validator->errors()
                ], Response::HTTP_BAD_REQUEST);
            }

            $data = $validator->validated();

            // Handle Logo Upload (file) - same as original
            if ($request->hasFile('Org_Logo')) {
                $file = $request->file('Org_Logo');
                $imageName = time() . '_' . uniqid() . '.' . $file->getClientOriginalExtension();

                $file->move(public_path('public/organisation/logo'), $imageName);

                // Store FULL public URL
                $data['Org_Logo'] = url('public/organisation/logo/' . $imageName);
            }

            $orgMaster = OrgMaster::create($data);

            return response()->json([
                'success' => true,
                'message' => 'Organisation created successfully',
                'data' => $orgMaster
            ], Response::HTTP_CREATED);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to create organisation',
                'error' => $e->getMessage()
            ], Response::HTTP_INTERNAL_SERVER_ERROR);
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        try {
            $orgMaster = OrgMaster::find($id);

            if (!$orgMaster) {
                return response()->json([
                    'success' => false,
                    'message' => 'Organisation not found'
                ], Response::HTTP_NOT_FOUND);
            }

            return response()->json([
                'success' => true,
                'message' => 'Organisation retrieved successfully',
                'data' => $orgMaster
            ], Response::HTTP_OK);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to retrieve organisation',
                'error' => $e->getMessage()
            ], Response::HTTP_INTERNAL_SERVER_ERROR);
        }
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        try {
            $existingCount = OrgMaster::count();

            // If no record exists, create first
            if ($existingCount === 0) {
                return $this->createFirstOrganisation($request);
            }

            $orgMaster = OrgMaster::first();

            if ($orgMaster->Org_Id != $id) {
                return response()->json([
                    'success' => false,
                    'message' => 'Only one organisation is allowed. Update the existing one.',
                    'existing_id' => $orgMaster->Org_Id
                ], Response::HTTP_BAD_REQUEST);
            }

            $validator = Validator::make($request->all(), OrgMaster::updateRules($orgMaster->Org_Id));

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation error',
                    'errors' => $validator->errors()
                ], Response::HTTP_BAD_REQUEST);
            }

            $data = $validator->validated();

            // Handle Logo Update - same as original
            if ($request->filled('Org_Logo')) {

                // Delete old image if exists
                if ($orgMaster->Org_Logo) {
                    $oldImagePath = public_path(str_replace(url('/') . '/', '', $orgMaster->Org_Logo));
                    if (file_exists($oldImagePath)) {
                        unlink($oldImagePath);
                    }
                }

                $imageName = null;

                // Base64 image
                if (str_contains($request->Org_Logo, 'base64')) {
                    $base64 = preg_replace('#^data:image/\w+;base64,#i', '', $request->Org_Logo);
                    $image = base64_decode($base64);

                    $imageName = 'org_' . time() . '_' . uniqid() . '.png';

                    $folder = public_path('organisation/logo');

                    if (!file_exists($folder)) {
                        mkdir($folder, 0755, true);
                    }

                    file_put_contents($folder . '/' . $imageName, $image);
                }
                // File upload
                elseif ($request->hasFile('Org_Logo')) {
                    $file = $request->file('Org_Logo');
                    $imageName = 'org_' . time() . '_' . uniqid() . '.' . $file->getClientOriginalExtension();

                    $folder = public_path('organisation/logo');
                    if (!file_exists($folder)) {
                        mkdir($folder, 0755, true);
                    }

                    $file->move($folder, $imageName);
                }

                if ($imageName) {
                    $data['Org_Logo'] = url('organisation/logo/' . $imageName);
                }
            }

            $orgMaster->update($data);

            return response()->json([
                'success' => true,
                'message' => 'Organisation updated successfully',
                'data' => $orgMaster,
                'action' => 'updated'
            ], Response::HTTP_OK);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to process organisation',
                'error' => $e->getMessage()
            ], Response::HTTP_INTERNAL_SERVER_ERROR);
        }
    }

    /**
     * Create the first (and only) organisation
     */
    private function createFirstOrganisation(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), OrgMaster::createRules());

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation error',
                    'errors' => $validator->errors()
                ], Response::HTTP_BAD_REQUEST);
            }

            $data = $validator->validated();

            // Handle Logo Upload - same as original store logic
            if ($request->hasFile('Org_Logo')) {
                $file = $request->file('Org_Logo');
                $imageName = time() . '_' . uniqid() . '.' . $file->getClientOriginalExtension();

                $file->move(public_path('public/organisation/logo'), $imageName);

                $data['Org_Logo'] = url('public/organisation/logo/' . $imageName);
            }

            $orgMaster = OrgMaster::create($data);

            return response()->json([
                'success' => true,
                'message' => 'First organisation created successfully',
                'data' => $orgMaster,
                'action' => 'created'
            ], Response::HTTP_CREATED);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to create organisation',
                'error' => $e->getMessage()
            ], Response::HTTP_INTERNAL_SERVER_ERROR);
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        try {
            $orgMaster = OrgMaster::find($id);

            if (!$orgMaster) {
                return response()->json([
                    'success' => false,
                    'message' => 'Organisation not found'
                ], Response::HTTP_NOT_FOUND);
            }

            $orgMaster->delete();

            return response()->json([
                'success' => true,
                'message' => 'Organisation deleted successfully'
            ], Response::HTTP_OK);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to delete organisation',
                'error' => $e->getMessage()
            ], Response::HTTP_INTERNAL_SERVER_ERROR);
        }
    }

    /**
     * Restore the specified soft deleted resource.
     */
    public function restore(string $id)
    {
        try {
            $orgMaster = OrgMaster::withTrashed()->where('Org_Id', $id)->first();

            if (!$orgMaster) {
                return response()->json([
                    'success' => false,
                    'message' => 'Organisation not found'
                ], Response::HTTP_NOT_FOUND);
            }

            if (!$orgMaster->trashed()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Organisation is not deleted'
                ], Response::HTTP_BAD_REQUEST);
            }

            $orgMaster->restore();

            return response()->json([
                'success' => true,
                'message' => 'Organisation restored successfully',
                'data' => $orgMaster
            ], Response::HTTP_OK);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to restore organisation',
                'error' => $e->getMessage()
            ], Response::HTTP_INTERNAL_SERVER_ERROR);
        }
    }

    /**
     * Permanently delete the specified resource.
     */
    public function forceDelete(string $id)
    {
        try {
            $orgMaster = OrgMaster::withTrashed()->where('Org_Id', $id)->first();

            if (!$orgMaster) {
                return response()->json([
                    'success' => false,
                    'message' => 'Organisation not found'
                ], Response::HTTP_NOT_FOUND);
            }

            $orgMaster->forceDelete();

            return response()->json([
                'success' => true,
                'message' => 'Organisation permanently deleted successfully'
            ], Response::HTTP_OK);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to permanently delete organisation',
                'error' => $e->getMessage()
            ], Response::HTTP_INTERNAL_SERVER_ERROR);
        }
    }

    /**
     * Get all trashed organisations.
     */
    public function trashed()
    {
        try {
            $trashed = OrgMaster::onlyTrashed()->get();

            return response()->json([
                'success' => true,
                'message' => 'Deleted organisations retrieved successfully',
                'data' => $trashed
            ], Response::HTTP_OK);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to retrieve deleted organisations',
                'error' => $e->getMessage()
            ], Response::HTTP_INTERNAL_SERVER_ERROR);
        }
    }
}
