<?php

namespace App\Http\Controllers;

use App\Models\PayBandMaster;
use App\Models\PayCommissionMaster;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class PayBandMasterController extends Controller
{
    public function index(Request $request)
    {
        try {
            $query = PayBandMaster::with('commission');

            $perPage = $request->get('per_page', 25);

            if ($perPage === 'all') {
                $data = $query->get();
            } else {
                $data = $query->paginate((int) $perPage);
            }

            return response()->json([
                'success' => true,
                'data' => $data
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => $e->getMessage()
            ], 500);
        }
    }
    public function all()
    {
        try {
            $data = PayBandMaster::with('commission')->get();

            return response()->json([
                'success' => true,
                'data' => $data
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => $e->getMessage()
            ], 500);
        }
    }


    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'PC_Id'   => 'required|exists:pay_comm_master,PC_Id',
            'PB_Code' => 'required|string|max:20',
            'Min_Pay' => 'required|numeric|min:0',
            'Max_Pay' => 'required|numeric|min:0|gt:Min_Pay'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors()
            ], 422);
        }

        $payBand = PayBandMaster::create($validator->validated());

        return response()->json([
            'success' => true,
            'message' => 'Pay Band created successfully',
            'data' => $payBand->load('commission')
        ], 201);
    }

    public function show(string $id)
    {
        $payBand = PayBandMaster::with('commission')->find($id);

        if (!$payBand) {
            return response()->json([
                'success' => false,
                'message' => 'Pay Band not found'
            ], 404);
        }

        return response()->json([
            'success' => true,
            'data' => $payBand
        ]);
    }

    public function update(Request $request, string $id)
    {
        $payBand = PayBandMaster::find($id);

        if (!$payBand) {
            return response()->json([
                'success' => false,
                'message' => 'Pay Band not found'
            ], 404);
        }

        $validator = Validator::make($request->all(), [
            'PC_Id'   => 'exists:pay_comm_master,PC_Id',
            'PB_Code' => 'string|max:20',
            'Min_Pay' => 'numeric|min:0',
            'Max_Pay' => 'numeric|min:0'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors()
            ], 422);
        }

        $payBand->update($validator->validated());

        return response()->json([
            'success' => true,
            'message' => 'Pay Band updated successfully',
            'data' => $payBand->load('commission')
        ]);
    }

    public function destroy(string $id)
    {
        $payBand = PayBandMaster::find($id);

        if (!$payBand) {
            return response()->json([
                'success' => false,
                'message' => 'Pay Band not found'
            ], 404);
        }

        $payBand->delete();

        return response()->json([
            'success' => true,
            'message' => 'Pay Band deleted successfully'
        ]);
    }

    public function getByCommission(string $commissionId)
    {
        $commission = PayCommissionMaster::find($commissionId);

        if (!$commission) {
            return response()->json([
                'success' => false,
                'message' => 'Commission not found'
            ], 404);
        }

        $payBands = PayBandMaster::where('PC_Id', $commissionId)->get();

        return response()->json([
            'success' => true,
            'data' => $payBands
        ]);
    }
}
