<?php

namespace App\Http\Controllers;

use App\Models\PayScaleMaster;
use App\Models\PayCommissionMaster;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class PayScaleMasterController extends Controller
{
    public function index(Request $request)
    {
        try {
            $query = PayScaleMaster::with('commission');

            $perPage = $request->get('per_page', 25);

            if ($perPage === 'all') {
                $data = $query->get();
            } else {
                $data = $query->paginate((int) $perPage);
            }

            return response()->json([
                'success' => true,
                'data' => $data
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => $e->getMessage()
            ], 500);
        }
    }

    public function all()
    {
        try {
            $data = PayScaleMaster::with('commission')->get();

            return response()->json([
                'success' => true,
                'data' => $data
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => $e->getMessage()
            ], 500);
        }
    }

    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'PC_Id'     => 'required|exists:pay_comm_master,PC_Id',
            'Scl_Code'  => 'required|string|max:20',
            'St_Basic'  => 'required|numeric|min:0',
            'Inc_Amount' => 'required|numeric|min:0',
            'Inc_Steps' => 'required|integer|min:1',
            'Max_Basic' => 'required|numeric|min:0'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors()
            ], 422);
        }

        $payScale = PayScaleMaster::create($validator->validated());

        return response()->json([
            'success' => true,
            'message' => 'Pay Scale created successfully',
            'data' => $payScale->load('commission')
        ], 201);
    }

    public function show(string $id)
    {
        $payScale = PayScaleMaster::with('commission')->find($id);

        if (!$payScale) {
            return response()->json([
                'success' => false,
                'message' => 'Pay Scale not found'
            ], 404);
        }

        return response()->json([
            'success' => true,
            'data' => $payScale
        ]);
    }

    public function update(Request $request, string $id)
    {
        $payScale = PayScaleMaster::find($id);

        if (!$payScale) {
            return response()->json([
                'success' => false,
                'message' => 'Pay Scale not found'
            ], 404);
        }

        $validator = Validator::make($request->all(), [
            'PC_Id'      => 'exists:pay_comm_master,PC_Id',
            'Scl_Code'   => 'string|max:20',
            'St_Basic'   => 'numeric|min:0',
            'Inc_Amount' => 'numeric|min:0',
            'Inc_Steps'  => 'integer|min:1',
            'Max_Basic'  => 'numeric|min:0'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors()
            ], 422);
        }

        $payScale->update($validator->validated());

        return response()->json([
            'success' => true,
            'message' => 'Pay Scale updated successfully',
            'data' => $payScale->load('commission')
        ]);
    }

    public function destroy(string $id)
    {
        $payScale = PayScaleMaster::find($id);

        if (!$payScale) {
            return response()->json([
                'success' => false,
                'message' => 'Pay Scale not found'
            ], 404);
        }

        $payScale->delete();

        return response()->json([
            'success' => true,
            'message' => 'Pay Scale deleted successfully'
        ]);
    }

    public function getByCommission(string $commissionId)
    {
        $commission = PayCommissionMaster::find($commissionId);

        if (!$commission) {
            return response()->json([
                'success' => false,
                'message' => 'Commission not found'
            ], 404);
        }

        $payScales = PayScaleMaster::where('PC_Id', $commissionId)->get();

        return response()->json([
            'success' => true,
            'data' => $payScales
        ]);
    }
}
