<?php

namespace App\Http\Controllers;

use App\Models\RelationshipMaster;
use Illuminate\Http\Request;
use Illuminate\Validation\ValidationException;
use Illuminate\Support\Facades\Log;
use Throwable;

class RelationshipMasterController extends Controller
{
    public function index(Request $request)
    {
        try {
            $queryParam = $request->input('query');

            $relationships = RelationshipMaster::query();

            // Search Filter
            if ($queryParam) {
                $relationships->where(function ($q) use ($queryParam) {
                    $q->where('Rela_Code', 'like', '%' . $queryParam . '%')
                        ->orWhere('Rela_Name', 'like', '%' . $queryParam . '%');
                });
            }

            $relationships->orderBy('Rela_Id', 'desc');

            // Pagination
            $perPage = $request->get('per_page', 25);

            if ($perPage === 'all') {
                $data = $relationships->get();
            } else {
                $data = $relationships->paginate((int) $perPage);
            }

            return response()->json([
                'status' => 'success',
                'message' => 'Relationships fetched successfully',
                'data' => $data
            ]);
        } catch (Throwable $e) {
            Log::error('Relationship Fetch Error: ' . $e->getMessage());

            return response()->json([
                'status' => 'error',
                'message' => 'Unable to fetch relationships'
            ], 500);
        }
    }

    // LIST ALL (NO PAGINATION)
    public function all()
    {
        try {
            $data = RelationshipMaster::orderBy('Rela_Id', 'desc')->get();

            return response()->json([
                'status' => 'success',
                'message' => 'Relationships fetched successfully',
                'data' => $data
            ]);
        } catch (Throwable $e) {
            Log::error('Relationship Fetch All Error: ' . $e->getMessage());

            return response()->json([
                'status' => 'error',
                'message' => 'Unable to fetch relationships'
            ], 500);
        }
    }

    public function store(Request $request)
    {
        try {
            $validated = $request->validate([
                'Rela_Code' => 'required|string|max:20|unique:relationship_master,Rela_Code',
                'Rela_Name' => 'required|string|max:50',
            ]);

            $relationship = RelationshipMaster::create($validated);

            return response()->json([
                'status' => 'success',
                'message' => 'Relationship created successfully',
                'data' => $relationship
            ], 201);
        } catch (ValidationException $e) {
            return response()->json([
                'status' => 'error',
                'errors' => $e->errors()
            ], 422);
        } catch (Throwable $e) {
            Log::error('Relationship Create Error: ' . $e->getMessage());
            return response()->json([
                'status' => 'error',
                'message' => 'Unable to create relationship'
            ], 500);
        }
    }

    public function show($id)
    {
        $relationship = RelationshipMaster::find($id);

        if (!$relationship) {
            return response()->json([
                'status' => 'error',
                'message' => 'Relationship not found'
            ], 404);
        }

        return response()->json([
            'status' => 'success',
            'data' => $relationship
        ]);
    }

    public function update(Request $request, $id)
    {
        try {
            $relationship = RelationshipMaster::find($id);

            if (!$relationship) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Relationship not found'
                ], 404);
            }

            $validated = $request->validate([
                'Rela_Code' => 'sometimes|required|string|max:20|unique:relationship_master,Rela_Code,' . $id . ',Rela_Id',
                'Rela_Name' => 'sometimes|required|string|max:50',
            ]);

            $relationship->update($validated);

            return response()->json([
                'status' => 'success',
                'message' => 'Relationship updated successfully',
                'data' => $relationship
            ]);
        } catch (ValidationException $e) {
            return response()->json([
                'status' => 'error',
                'errors' => $e->errors()
            ], 422);
        } catch (Throwable $e) {
            Log::error('Relationship Update Error: ' . $e->getMessage());
            return response()->json([
                'status' => 'error',
                'message' => 'Unable to update relationship'
            ], 500);
        }
    }

    public function destroy($id)
    {
        $relationship = RelationshipMaster::find($id);

        if (!$relationship) {
            return response()->json([
                'status' => 'error',
                'message' => 'Relationship not found'
            ], 404);
        }

        $relationship->delete();

        return response()->json([
            'status' => 'success',
            'message' => 'Relationship deleted successfully'
        ]);
    }
}
