<?php

namespace App\Http\Controllers;

use App\Models\SchemeMaster;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Throwable;

class SchemeMasterController extends Controller
{
    // 🔹 INDEX
   public function index(Request $request)
{
    try {

        $perPage = $request->get('per_page', 25);
        $search = $request->get('search', '');
        $withTrashed = $request->get('with_trashed', false);

        $query = SchemeMaster::with(['department', 'grantType'])
                    ->orderBy('Sch_id', 'desc');

        // Include soft deleted if requested
        if ($withTrashed) {
            $query->withTrashed();
        }

        // Search filter
        if ($search) {
            $query->where(function ($q) use ($search) {
                $q->where('Sch_Name', 'like', "%{$search}%")
                  ->orWhere('Sch_Code', 'like', "%{$search}%")
                  ->orWhere('Sch_id', 'like', "%{$search}%");
            });
        }

        // Pagination or all
        if ($perPage === 'all') {
            $data = $query->get();
        } else {
            $data = $query->paginate((int) $perPage);
        }

        return response()->json([
            'success' => true,
            'message' => 'Schemes fetched successfully',
            'data' => $data
        ], 200);

    } catch (\Throwable $e) {
        return response()->json([
            'success' => false,
            'message' => 'Something went wrong',
            'error' => $e->getMessage()
        ], 500);
    }
}
public function all()
{
    try {
        $data = SchemeMaster::with(['department', 'grantType'])
                    ->orderBy('Sch_id', 'desc')
                    ->get();

        return response()->json([
            'success' => true,
            'message' => 'Schemes fetched successfully',
            'data' => $data
        ], 200);

    } catch (\Throwable $e) {
        return response()->json([
            'success' => false,
            'message' => 'Something went wrong',
            'error' => $e->getMessage()
        ], 500);
    }
}


    // 🔹 STORE
    public function store(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'Sch_Code' => 'required|string|max:30|unique:scheme_master,Sch_Code',
                'Sch_Name' => 'required|string|max:50',
                'Dept_Id'  => 'nullable|exists:dept_master,Dpt_id',
                'GT_Id'    => 'nullable|exists:grant_type_master,GT_Id',
                'St_Date'  => 'nullable|date',
                'Ed_Date'  => 'nullable|date|after_or_equal:St_Date',
                'Budget'   => 'nullable|numeric|min:0',
                'Status'   => 'nullable|string|max:15',
                'Is_Active'=> 'nullable|boolean'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'errors' => $validator->errors()
                ], 422);
            }

            $scheme = SchemeMaster::create([
                'Sch_Code' => $request->Sch_Code,
                'Sch_Name' => $request->Sch_Name,
                'Dept_Id'  => $request->Dept_Id,
                'GT_Id'    => $request->GT_Id,
                'St_Date'  => $request->St_Date,
                'Ed_Date'  => $request->Ed_Date,
                'Budget'   => $request->Budget,
                'Status'   => $request->Status ?? 'ACTIVE',
                'Is_Active'=> $request->Is_Active ?? true,
            ]);

            $scheme->load(['department', 'grantType']);

            return response()->json([
                'success' => true,
                'message' => 'Scheme created successfully',
                'data' => $scheme
            ], 201);

        } catch (Throwable $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to create scheme',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    // 🔹 SHOW
    public function show($id)
    {
        try {
            $scheme = SchemeMaster::with(['department', 'grantType'])->find($id);

            if (!$scheme) {
                return response()->json([
                    'success' => false,
                    'message' => 'Scheme not found'
                ], 404);
            }

            return response()->json([
                'success' => true,
                'data' => $scheme
            ], 200);

        } catch (Throwable $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error fetching scheme',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    // 🔹 UPDATE
    public function update(Request $request, $id)
    {
        try {
            $scheme = SchemeMaster::find($id);

            if (!$scheme) {
                return response()->json([
                    'success' => false,
                    'message' => 'Scheme not found'
                ], 404);
            }

            $validator = Validator::make($request->all(), [
                'Sch_Code' => 'sometimes|required|string|max:30|unique:scheme_master,Sch_Code,' . $id . ',Sch_id',
                'Sch_Name' => 'sometimes|required|string|max:50',
                'Dept_Id'  => 'sometimes|nullable|exists:dept_master,Dpt_id',
                'GT_Id'    => 'sometimes|nullable|exists:grant_type_master,GT_Id',
                'St_Date'  => 'sometimes|nullable|date',
                'Ed_Date'  => 'sometimes|nullable|date|after_or_equal:St_Date',
                'Budget'   => 'sometimes|nullable|numeric|min:0',
                'Status'   => 'sometimes|nullable|string|max:15',
                'Is_Active'=> 'sometimes|boolean'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'errors' => $validator->errors()
                ], 422);
            }

            $scheme->update($request->all());
            $scheme->load(['department', 'grantType']);

            return response()->json([
                'success' => true,
                'message' => 'Scheme updated successfully',
                'data' => $scheme
            ], 200);

        } catch (Throwable $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to update scheme',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    // 🔹 DELETE (Soft Delete)
    public function destroy($id)
    {
        try {
            $scheme = SchemeMaster::find($id);

            if (!$scheme) {
                return response()->json([
                    'success' => false,
                    'message' => 'Scheme not found'
                ], 404);
            }

            $scheme->delete();

            return response()->json([
                'success' => true,
                'message' => 'Scheme deleted successfully'
            ], 200);

        } catch (Throwable $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to delete scheme',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    public function restore($id)
    {
        $schemeType = SchemeMaster::find($id);

        if (!$schemeType) {
            return response()->json([
                'success' => false,
                'message' => 'Scheme type not found'
            ], 404);
        }

        $schemeType->update(['is_active' => true]);

        return response()->json([
            'success' => true,
            'message' => 'Scheme type activated successfully',
            'data' => $schemeType
        ]);
    }
}
