<?php

namespace App\Http\Controllers;

use App\Http\Controllers\Controller;
use App\Models\VendInvParameter;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class VendInvParameterController extends Controller
{

    /**
     * List + Search + Pagination
     */
public function index(Request $request)
{
    try {

        $query = VendInvParameter::query();

        // Search
        if ($request->search) {
            $search = $request->search;

            $query->where(function ($q) use ($search) {
                $q->where('Formula', 'like', "%$search%")
                  ->orWhere('VIP_Sr', 'like', "%$search%");
            });
        }

        if ($request->TnT_ID) {
            $query->where('TnT_ID', $request->TnT_ID);
        }

        if ($request->Eff_Date) {
            $query->whereDate('Eff_Date', $request->Eff_Date);
        }

        $perPage = $request->get('per_page', 10);

        // Group Pagination
        $groups = $query->select('TnT_ID', 'Eff_Date')
            ->distinct()
            ->orderBy('TnT_ID')
            ->orderBy('Eff_Date', 'desc')
            ->paginate($perPage);

        $groupedData = [];

        foreach ($groups as $group) {

            $items = VendInvParameter::where('TnT_ID', $group->TnT_ID)
                ->whereDate('Eff_Date', $group->Eff_Date)
                ->with(['TendorType', 'ledger', 'InvoiceCharge'])
                ->orderBy('VIP_Sr', 'asc')
                ->get();

            if ($items->isEmpty()) {
                continue;
            }

            $key = $group->TnT_ID . '|' . $group->Eff_Date;

         $groupedData[$key] = [
    'tnt_id' => $group->TnT_ID,
    'eff_date' => $group->Eff_Date,
    'formatted_eff_date' => \Carbon\Carbon::parse($group->Eff_Date)->format('d-m-Y'),
    'tender' => $items->first()->TendorType,   // ✅ correct relation
    'parameters' => []
];

   foreach ($items as $item) {

    $groupedData[$key]['parameters'][] = [
        'VIP_Id' => $item->VIP_Id,
        'VIP_Sr' => $item->VIP_Sr,
        'IC_Id' => $item->IC_Id,
        'Add_Dedu' => $item->Add_Dedu,
        'add_deduct_text' => $item->add_deduct_text,
        'Lg_Id' => $item->Lg_Id,
        'Percent' => $item->Percent,
        'Formula' => $item->Formula,

        'ledger' => $item->ledger,
        'invoice_charge' => $item->InvoiceCharge,   // ✅ add this
    ];
}
        }

        return response()->json([
            'success' => true,
            'message' => 'VendInv Parameters retrieved successfully',
            'data' => array_values($groupedData),
            'pagination' => [
                'current_page' => $groups->currentPage(),
                'per_page' => $groups->perPage(),
                'total' => $groups->total(),
                'last_page' => $groups->lastPage(),
            ]
        ]);

    } catch (\Exception $e) {

        return response()->json([
            'success' => false,
            'message' => 'Failed to retrieve records',
            'error' => $e->getMessage()
        ], 500);
    }
}



    /**
 * Get All Records (No Pagination)
 */
public function all()
{
    $items = VendInvParameter::with(['TendorType', 'ledger', 'InvoiceCharge'])
        ->orderBy('VIP_Sr')
        ->get();

    $groupedData = [];

    foreach ($items as $item) {

        $key = $item->TnT_ID . '|' . $item->Eff_Date->format('Y-m-d');

        if (!isset($groupedData[$key])) {
            $groupedData[$key] = [
                'tnt_id' => $item->TnT_ID,
                'eff_date' => $item->Eff_Date,
                'formatted_eff_date' => $item->Eff_Date->format('d-m-Y'),
                'tender' => $item->TendorType,
                'parameters' => []
            ];
        }

        $groupedData[$key]['parameters'][] = [
            'VIP_Id' => $item->VIP_Id,
            'VIP_Sr' => $item->VIP_Sr,
            'IC_Id' => $item->IC_Id,
            'Add_Dedu' => $item->Add_Dedu,
            'add_deduct_text' => $item->add_deduct_text,
            'Lg_Id' => $item->Lg_Id,
            'Percent' => $item->Percent,
            'Formula' => $item->Formula,
            'ledger' => $item->ledger,
            'item_category' => $item->itemCategory
        ];
    }

    return response()->json([
        'success' => true,
        'message' => 'All VendInv Parameters',
        'data' => array_values($groupedData)
    ]);
}


    /**
     * Store
     */
public function store(Request $request)
{
    try {

        $validator = Validator::make($request->all(), [

            'TnT_ID' => 'required|integer',
            'Eff_Date' => 'required|date',

            'parameters' => 'required|array|min:1',

            'parameters.*.VIP_Sr' => 'required|integer',
            'parameters.*.IC_Id' => 'required|integer',
            'parameters.*.Add_Dedu' => 'required|in:0,1',
            'parameters.*.Lg_Id' => 'required|integer',
            'parameters.*.Percent' => 'required|numeric',
            'parameters.*.Formula' => 'nullable|string|max:100',

        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => $validator->errors()
            ], 422);
        }

        $insertedData = [];

        foreach ($request->parameters as $param) {

            $data = VendInvParameter::create([
                'TnT_ID'   => $request->TnT_ID,
                'Eff_Date' => $request->Eff_Date,
                'VIP_Sr'   => $param['VIP_Sr'],
                'IC_Id'    => $param['IC_Id'],
                'Add_Dedu' => $param['Add_Dedu'],
                'Lg_Id'    => $param['Lg_Id'],
                'Percent'  => $param['Percent'],
                'Formula'  => $param['Formula'] ?? null,
            ]);

            $insertedData[] = $data;
        }

        return response()->json([
            'success' => true,
            'message' => 'Records Created Successfully',
            'data' => $insertedData
        ]);

    } catch (\Exception $e) {

        return response()->json([
            'success' => false,
            'message' => 'Failed to create records',
            'error' => $e->getMessage()
        ], 500);
    }
}


    public function show($id)
    {
        $data = VendInvParameter::with(['TendorType', 'ledger', 'InvoiceCharge'])
                    ->find($id);

        if (!$data) {
            return response()->json([
                'success' => false,
                'message' => 'Record Not Found'
            ], 404);
        }

        return response()->json([
            'success' => true,
            'data' => $data
        ]);
    }

public function showByTender($tntId)
{
    $items = VendInvParameter::with(['TendorType', 'ledger', 'InvoiceCharge'])
        ->where('TnT_ID', $tntId)
        ->orderBy('Eff_Date', 'desc')
        ->orderBy('VIP_Sr', 'asc')
        ->get();

    if ($items->isEmpty()) {
        return response()->json([
            'success' => false,
            'message' => 'Record Not Found'
        ], 404);
    }

    $groupedData = [];

    foreach ($items as $item) {

        $key = $item->TnT_ID . '|' . $item->Eff_Date->format('Y-m-d');

        if (!isset($groupedData[$key])) {
            $groupedData[$key] = [
                'tnt_id' => $item->TnT_ID,
                'eff_date' => $item->Eff_Date,
                'formatted_eff_date' => $item->Eff_Date->format('d-m-Y'),
                'tender' => $item->TendorType,
                'parameters' => []
            ];
        }

        $groupedData[$key]['parameters'][] = [
            'VIP_Id' => $item->VIP_Id,
            'VIP_Sr' => $item->VIP_Sr,
            'IC_Id' => $item->IC_Id,
            'Add_Dedu' => $item->Add_Dedu,
            'add_deduct_text' => $item->add_deduct_text,
            'Lg_Id' => $item->Lg_Id,
            'Percent' => $item->Percent,
            'Formula' => $item->Formula,
            'ledger' => $item->ledger,
            'invoice_charge' => $item->InvoiceCharge
        ];
    }

    return response()->json([
        'success' => true,
        'message' => 'Records retrieved successfully',
        'data' => array_values($groupedData)
    ]);
}


    public function update(Request $request, $id)
    {
        $data = VendInvParameter::find($id);

        if (!$data) {
            return response()->json([
                'success' => false,
                'message' => 'Record Not Found'
            ], 404);
        }

        $validator = Validator::make($request->all(), [

            'TnT_ID' => 'sometimes|integer',
            'Eff_Date' => 'sometimes|date',
            'VIP_Sr' => 'sometimes|integer',
            'IC_Id' => 'sometimes|integer',
            'Add_Dedu' => 'sometimes|in:0,1',
            'Lg_Id' => 'sometimes|integer',
            'Percent' => 'sometimes|numeric',
            'Formula' => 'nullable|string|max:100',

        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => $validator->errors()
            ], 422);
        }

        $data->update($request->all());

        return response()->json([
            'success' => true,
            'message' => 'Record Updated Successfully',
            'data' => $data
        ]);
    }

    /**
     * Soft Delete
     */
    public function destroy($id)
    {
        $data = VendInvParameter::find($id);

        if (!$data) {
            return response()->json([
                'success' => false,
                'message' => 'Record Not Found'
            ], 404);
        }

        $data->delete();

        return response()->json([
            'success' => true,
            'message' => 'Record Deleted Successfully'
        ]);
    }
}
