<?php

namespace App\Http\Controllers;

use App\Models\WardMaster;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class WardMasterController extends Controller
{
    public function index(Request $request)
    {
        $query = WardMaster::with(['zone', 'zone.organisation'])
                    ->where('is_active', true);

        if ($request->has('Zon_Id')) {
            $query->where('Zon_Id', $request->Zon_Id);
        }

        if ($request->has('search')) {
            $query->search($request->search);
        }

        $perPage = $request->get('per_page', 15);
        $wards = $query->paginate($perPage);

        return response()->json([
            'success' => true,
            'data' => $wards,
            'message' => 'Wards retrieved successfully'
        ]);
    }
    
    public function all(Request $request)
{
    $query = WardMaster::with(['zone', 'zone.organisation']);

    // Optional Filters
    if ($request->has('Zon_Id') && !empty($request->Zon_Id)) {
        $query->where('Zon_Id', $request->Zon_Id);
    }

    if ($request->has('is_active')) {
        $query->where('is_active', $request->is_active);
    }

    if ($request->has('search') && !empty($request->search)) {
        $query->search($request->search);
    }

    // Sorting
    $sortField = $request->get('sort_by', 'Wrd_Id');
    $sortDirection = strtolower($request->get('sort_dir', 'desc')) === 'asc' ? 'asc' : 'desc';

    $allowedSortFields = [
        'Wrd_Id',
        'Wrd_Code',
        'Wrd_Name',
        'Zon_Id',
        'created_at',
        'updated_at'
    ];

    if (!in_array($sortField, $allowedSortFields)) {
        $sortField = 'Wrd_Id';
    }

    $wards = $query->orderBy($sortField, $sortDirection)->get();

    return response()->json([
        'success' => true,
        'message' => 'All wards retrieved successfully',
        'data' => $wards
    ], 200);
}

    public function store(Request $request)
    {
        $validator = Validator::make(
            $request->all(),
            WardMaster::createRules(),
            WardMaster::validationMessages()
        );

        $validator->after(function ($validator) use ($request) {
            if (!WardMaster::validateUniqueWardCode(
                $request->Wrd_Code,
                $request->Zon_Id
            )) {
                $validator->errors()->add('Wrd_Code', 'This ward code already exists in the selected zone');
            }
        });

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors()
            ], 422);
        }

        $ward = WardMaster::create([
            'Wrd_Code' => $request->Wrd_Code,
            'Wrd_Name' => $request->Wrd_Name,
            'Zon_Id' => $request->Zon_Id,
            'is_active' => true
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Ward created successfully',
            'data' => $ward->load('zone')
        ], 201);
    }

    public function show($id)
    {
        $ward = WardMaster::with(['zone'])->find($id);

        if (!$ward) {
            return response()->json([
                'success' => false,
                'message' => 'Ward not found'
            ], 404);
        }

        return response()->json([
            'success' => true,
            'data' => $ward,
            'message' => 'Ward retrieved successfully'
        ]);
    }

    public function update(Request $request, $id)
    {
        $ward = WardMaster::find($id);

        if (!$ward) {
            return response()->json([
                'success' => false,
                'message' => 'Ward not found'
            ], 404);
        }

        $validator = Validator::make(
            $request->all(),
            WardMaster::updateRules($id),
            WardMaster::validationMessages()
        );

        $validator->after(function ($validator) use ($request, $id) {
            if ($request->has('Wrd_Code') && $request->has('Zon_Id')) {
                if (!WardMaster::validateUniqueWardCode(
                    $request->Wrd_Code,
                    $request->Zon_Id,
                    $id
                )) {
                    $validator->errors()->add('Wrd_Code', 'This ward code already exists in the selected zone');
                }
            }
        });

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors()
            ], 422);
        }

        $ward->update($request->all());

        return response()->json([
            'success' => true,
            'message' => 'Ward updated successfully',
            'data' => $ward->load('zone')
        ]);
    }

    public function destroy($id)
    {
        $ward = WardMaster::find($id);

        if (!$ward) {
            return response()->json([
                'success' => false,
                'message' => 'Ward not found'
            ], 404);
        }

        if (!$ward->canBeDeleted()) {
            return response()->json([
                'success' => false,
                'message' => 'Cannot delete ward as it has dependent records'
            ], 400);
        }

        $ward->deactivate();

        return response()->json([
            'success' => true,
            'message' => 'Ward deactivated successfully'
        ]);
    }

    public function restore($id)
    {
        $ward = WardMaster::withTrashed()->find($id);

        if (!$ward) {
            return response()->json([
                'success' => false,
                'message' => 'Ward not found'
            ], 404);
        }

        $ward->activate();

        return response()->json([
            'success' => true,
            'message' => 'Ward activated successfully',
            'data' => $ward->load('zone')
        ]);
    }

    public function getByZone($zoneId)
    {
        $wards = WardMaster::with('zone')
                    ->where('Zon_Id', $zoneId)
                    ->where('is_active', true)
                    ->get();

        return response()->json([
            'success' => true,
            'data' => $wards,
            'message' => 'Wards retrieved successfully'
        ]);
    }
}
