<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class CccMaster extends Model
{
    use HasFactory, SoftDeletes;

    protected $table = 'ccc_master';
    protected $primaryKey = 'CCC_Id';

    protected $fillable = [
        'CCC_Code',
        'CCC_Name',
        'Wrd_Id',
        'is_active'
    ];

    protected $casts = [
        'CCC_Id' => 'integer',
        'Wrd_Id' => 'integer',
        'is_active' => 'boolean'
    ];

    /* ================= VALIDATION RULES ================= */

    public static function createRules()
    {
        return [
            'CCC_Code' => 'nullable|string|max:20|unique:ccc_master,CCC_Code',
            'CCC_Name' => 'required|string|max:50',
            'Wrd_Id'   => 'required|exists:ward_master,Wrd_Id'
        ];
    }

    public static function updateRules($id)
    {
        return [
            'CCC_Code' => 'sometimes|nullable|string|max:20|unique:ccc_master,CCC_Code,' . $id . ',CCC_Id',
            'CCC_Name' => 'sometimes|required|string|max:50',
            'Wrd_Id'   => 'sometimes|required|exists:ward_master,Wrd_Id',
            'is_active'=> 'sometimes|boolean'
        ];
    }

    public static function validationMessages()
    {
        return [
            'CCC_Name.required' => 'CCC name is required',
            'Wrd_Id.required'   => 'Ward is required',
            'Wrd_Id.exists'     => 'Selected ward does not exist'
        ];
    }

    /* ================= RELATIONSHIPS ================= */

    public function ward()
    {
        return $this->belongsTo(WardMaster::class, 'Wrd_Id', 'Wrd_Id');
    }

    public function zone()
    {
        return $this->hasOneThrough(
            ZoneMaster::class,
            WardMaster::class,
            'Wrd_Id',
            'Zone_Id',
            'Wrd_Id',
            'Zone_Id'
        );
    }

    /* ================= SCOPES ================= */

    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    public function scopeByWard($query, $wardId)
    {
        return $query->where('Wrd_Id', $wardId);
    }

    public function scopeSearch($query, $search)
    {
        return $query->where(function ($q) use ($search) {

            $q->where('CCC_Code', 'like', "%{$search}%")
              ->orWhere('CCC_Name', 'like', "%{$search}%")
              ->orWhereHas('ward', function ($q) use ($search) {

                  $q->where('Wrd_Code', 'like', "%{$search}%")
                    ->orWhere('Wrd_Name', 'like', "%{$search}%")
                    ->orWhereHas('zone', function ($q) use ($search) {

                        $q->where('Zone_Code', 'like', "%{$search}%")
                          ->orWhere('Zone_Name', 'like', "%{$search}%");

                    });
              });
        });
    }

    /* ================= ACCESSORS ================= */

    public function getWardNameAttribute()
    {
        return $this->ward ? $this->ward->Wrd_Name : null;
    }

    public function getWardCodeAttribute()
    {
        return $this->ward ? $this->ward->Wrd_Code : null;
    }

    public function getZoneNameAttribute()
    {
        return $this->ward && $this->ward->zone ? $this->ward->zone->Zone_Name : null;
    }

    public function getZoneCodeAttribute()
    {
        return $this->ward && $this->ward->zone ? $this->ward->zone->Zone_Code : null;
    }

    public function getFullCccAttribute()
    {
        return $this->CCC_Code . ' - ' . $this->CCC_Name;
    }

    /* ================= BUSINESS LOGIC ================= */

    public function canBeDeleted()
    {
        return true;
    }

    public function deactivate()
    {
        $this->update(['is_active' => false]);
        return $this;
    }

    public function activate()
    {
        $this->update(['is_active' => true]);
        return $this;
    }
}
