<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class GrantDeptMap extends Model
{
    use HasFactory, SoftDeletes;

    protected $table = 'grant_dept_map';
    protected $primaryKey = 'GDM_Id';

    protected $fillable = [
        'Dept_Id',
        'Lg_Id',          // JSON array of multiple ledger IDs
    ];

    protected $casts = [
        'Lg_Id' => 'array',  // JSON → array conversion
    ];

    /**
     * Department relation (belongsTo)
     */
    public function department()
    {
        return $this->belongsTo(Department::class, 'Dept_Id', 'Dpt_id');
    }

    /**
     * Get all related ledgers as collection
     */
    public function ledgers()
    {
        if (empty($this->Lg_Id)) {
            return collect();
        }

        return LedgerMaster::whereIn('Lg_Id', $this->Lg_Id)->get();
    }

    /**
     * Get first ledger (backward compatibility के लिए)
     */
    public function getFirstLedgerAttribute()
    {
        if (!empty($this->Lg_Id)) {
            return LedgerMaster::find($this->Lg_Id[0] ?? null);
        }
        return null;
    }

    // Ledger add/remove helpers (बहुत उपयोगी functions)
    public function addLedgerId($ledgerId)
    {
        $ledgerIds = $this->Lg_Id ?? [];

        if (!in_array($ledgerId, $ledgerIds)) {
            $ledgerIds[] = $ledgerId;
            $this->Lg_Id = $ledgerIds;
            return $this->save();
        }

        return false;
    }

    public function removeLedgerId($ledgerId)
    {
        $ledgerIds = $this->Lg_Id ?? [];
        $index = array_search($ledgerId, $ledgerIds);

        if ($index !== false) {
            unset($ledgerIds[$index]);
            $this->Lg_Id = array_values($ledgerIds);
            return $this->save();
        }

        return false;
    }

    public function hasLedgerId($ledgerId)
    {
        return in_array($ledgerId, $this->Lg_Id ?? []);
    }
}
