<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class LedgerMaster extends Model
{
    use HasFactory, SoftDeletes;

    protected $table = 'ledger_master';
    protected $primaryKey = 'Lg_Id';

    protected $fillable = [
        'Lg_Code',
        'Gr_Id',
        'Lg_Name',
        'Lg_Eng_Name',
        'Lg_Address',
        'Lg_City',
        'Lg_Pin',
        'Lg_Phone',
        'Lg_Email',
        'Lg_Sub',
        'Lg_Buk',
        'Lg_Cr_Lim',
        'Lg_Drw_Pwr',
        'Lg_Status'
    ];

    protected $casts = [
        'Lg_Cr_Lim' => 'decimal:2',
        'Lg_Drw_Pwr' => 'decimal:2',
        'Lg_Status' => 'boolean'
    ];

    /*
    |--------------------------------------------------------------------------
    | Relationships
    |--------------------------------------------------------------------------
    */

    public function ledgerGroup()
    {
        return $this->belongsTo(LedgGrp::class, 'Gr_Id', 'Gr_Id');
    }

    /*
    |--------------------------------------------------------------------------
    | Validation Rules
    |--------------------------------------------------------------------------
    */

    public static function createRules()
    {
        return [
            'Lg_Code' => 'nullable|string|max:50|unique:ledger_master,Lg_Code',
            'Gr_Id' => 'nullable|integer|exists:ledg_grps,Gr_Id',
            'Lg_Name' => 'required|string|max:100',
            'Lg_Eng_Name' => 'nullable|string|max:150',
            'Lg_Address' => 'nullable|string',
            'Lg_City' => 'nullable|string|max:50',
            'Lg_Pin' => 'nullable|string|max:20',
            'Lg_Phone' => 'nullable|string|max:15',
            'Lg_Email' => 'nullable|email|max:50',
            'Lg_Sub' => 'nullable|string|max:100',
            'Lg_Buk' => 'nullable|string|max:100',
            'Lg_Cr_Lim' => 'nullable|numeric|min:0',
            'Lg_Drw_Pwr' => 'nullable|numeric|min:0',
            'Lg_Status' => 'nullable|boolean'
        ];
    }

    public static function updateRules($id)
    {
        return [
            'Lg_Code' => 'nullable|string|max:50|unique:ledger_master,Lg_Code,' . $id . ',Lg_Id',
            'Gr_Id' => 'nullable|integer|exists:ledg_grps,Gr_Id',
            'Lg_Name' => 'sometimes|required|string|max:100',
            'Lg_Eng_Name' => 'nullable|string|max:150',
            'Lg_Address' => 'nullable|string',
            'Lg_City' => 'nullable|string|max:50',
            'Lg_Pin' => 'nullable|string|max:20',
            'Lg_Phone' => 'nullable|string|max:15',
            'Lg_Email' => 'nullable|email|max:50',
            'Lg_Sub' => 'nullable|string|max:100',
            'Lg_Buk' => 'nullable|string|max:100',
            'Lg_Cr_Lim' => 'nullable|numeric|min:0',
            'Lg_Drw_Pwr' => 'nullable|numeric|min:0',
            'Lg_Status' => 'nullable|boolean'
        ];
    }

    /*
    |--------------------------------------------------------------------------
    | Scopes
    |--------------------------------------------------------------------------
    */

    public function scopeActive($query)
    {
        return $query->where('Lg_Status', true);
    }

    public function scopeSearch($query, $search)
    {
        return $query->where(function ($q) use ($search) {
            $q->where('Lg_Code', 'like', "%{$search}%")
              ->orWhere('Lg_Name', 'like', "%{$search}%")
              ->orWhere('Lg_City', 'like', "%{$search}%")
              ->orWhere('Lg_Phone', 'like', "%{$search}%")
              ->orWhere('Lg_Email', 'like', "%{$search}%")
              ->orWhere('Lg_Eng_Name', 'like', "%{$search}%");
        });
    }

    /*
    |--------------------------------------------------------------------------
    | Business Logic
    |--------------------------------------------------------------------------
    */

    public function deactivate()
    {
        $this->update(['Lg_Status' => false]);
    }

    public function activate()
    {
        $this->update(['Lg_Status' => true]);
    }
}
