<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class SubLedgerMaster extends Model
{
    use HasFactory, SoftDeletes;

    protected $table = 'subledg_master';
    protected $primaryKey = 'SL_Id';

    protected $fillable = [
        'Lg_Id',
        'SL_Code',
        'SL_Name',
        'SL_Address',
        'SL_City',
        'SL_Pin',
        'SL_Phone',
        'SL_Email',
        'SL_Cont_Pers',
        'SL_Cr_Lim',
        'SL_Draw_Pwr',
        'SL_Status',
        // Statutory & Legal Details
        'PAN_Number',
        'PAN_Verified',
        'PAN_Verified_Date',
        'GSTIN',
        'GSTIN_Verified',
        'GSTIN_Verified_Date',
        'GST_Exempt_Declaration',
        'MSME_Udyam_Registration_No',
        'CIN_LLPIN',
        'Shop_Act_Trade_License_No',
        'TAN',
        // Banking & Payment Details
        'Bank_Name',
        'Branch_Name',
        'Account_Holder_Name',
        'Account_Number',
        'IFSC_Code',
        'Account_Type',
    ];

    protected $casts = [
        'Lg_Id' => 'array',
        'SL_Status' => 'boolean',
        'PAN_Verified' => 'boolean',
        'GSTIN_Verified' => 'boolean',
        'SL_Cr_Lim' => 'decimal:2',
        'SL_Draw_Pwr' => 'decimal:2',
        'PAN_Verified_Date' => 'date',
        'GSTIN_Verified_Date' => 'date',
    ];

    // Validation rules for create
    public static function createRules()
    {
        return [
            'Lg_Id' => 'required|array',
            'Lg_Id.*' => 'exists:ledger_master,Lg_Id',
            'SL_Code' => 'nullable|string|max:50|unique:subledg_master,SL_Code',
            'SL_Name' => 'required|string|max:150',
            'SL_Address' => 'nullable|string',
            'SL_City' => 'nullable|string|max:50',
            'SL_Pin' => 'nullable|string|max:20',
            'SL_Phone' => 'nullable|string|max:15',
            'SL_Email' => 'nullable|email|max:50',
            'SL_Cont_Pers' => 'nullable|string|max:50',
            'SL_Cr_Lim' => 'nullable|numeric|min:0|max:9999999999999.99',
            'SL_Draw_Pwr' => 'nullable|numeric|min:0|max:9999999999999.99',
            'SL_Status' => 'nullable|boolean',

            // Statutory & Legal Details validation
            'PAN_Number' => 'nullable|string|regex:/^[A-Z]{5}[0-9]{4}[A-Z]{1}$/|unique:subledg_master,PAN_Number',
            'GSTIN' => 'nullable|string|regex:/^[0-9]{2}[A-Z]{5}[0-9]{4}[A-Z]{1}[1-9A-Z]{1}Z[0-9A-Z]{1}$/|unique:subledg_master,GSTIN',
            'GST_Exempt_Declaration' => 'nullable|string|max:255',
            'MSME_Udyam_Registration_No' => 'nullable|string|max:19',
            'CIN_LLPIN' => 'nullable|string|max:21',
            'Shop_Act_Trade_License_No' => 'nullable|string|max:50',
            'TAN' => 'nullable|string|max:10',

            // Banking & Payment Details validation
            'Bank_Name' => 'nullable|string|max:50',
            'Branch_Name' => 'nullable|string|max:50',
            'Account_Holder_Name' => 'nullable|string|max:50',
            'Account_Number' => 'nullable|string|max:18',
            'IFSC_Code' => 'nullable|string|max:20',
            'Account_Type' => 'nullable|in:Savings,Current,OD,CC,NRI,Other',

            // Verification fields
            'PAN_Verified' => 'nullable|boolean',
            'GSTIN_Verified' => 'nullable|boolean',
        ];
    }

    // Validation rules for update
    public static function updateRules($id)
    {
        return [
            'Lg_Id' => 'sometimes|nullable|array|min:1',
            'Lg_Id.*' => 'exists:ledger_master,Lg_Id',
            'SL_Code' => 'sometimes|nullable|string|max:50|unique:subledg_master,SL_Code,' . $id . ',SL_Id',
            'SL_Name' => 'sometimes|nullable|string|max:150',
            'SL_Address' => 'nullable|string',
            'SL_City' => 'nullable|string|max:50',
            'SL_Pin' => 'nullable|string|max:20',
            'SL_Phone' => 'nullable|string|max:15',
            'SL_Email' => 'nullable|email|max:50',
            'SL_Cont_Pers' => 'nullable|string|max:50',
            'SL_Cr_Lim' => 'nullable|numeric|min:0|max:9999999999999.99',
            'SL_Draw_Pwr' => 'nullable|numeric|min:0|max:9999999999999.99',
            'SL_Status' => 'nullable|boolean',

            // Statutory & Legal Details validation
            'PAN_Number' => 'nullable|string|regex:/^[A-Z]{5}[0-9]{4}[A-Z]{1}$/|unique:subledg_master,PAN_Number,' . $id . ',SL_Id',
            'GSTIN' => 'nullable|string|regex:/^[0-9]{2}[A-Z]{5}[0-9]{4}[A-Z]{1}[1-9A-Z]{1}Z[0-9A-Z]{1}$/|unique:subledg_master,GSTIN,' . $id . ',SL_Id',
            'GST_Exempt_Declaration' => 'nullable|string|max:255',
            'MSME_Udyam_Registration_No' => 'nullable|string|max:19',
            'CIN_LLPIN' => 'nullable|string|max:21',
            'Shop_Act_Trade_License_No' => 'nullable|string|max:50',
            'TAN' => 'nullable|string|max:10',

            // Banking & Payment Details validation
            'Bank_Name' => 'nullable|string|max:50',
            'Branch_Name' => 'nullable|string|max:50',
            'Account_Holder_Name' => 'nullable|string|max:50',
            'Account_Number' => 'nullable|string|max:18',
            'IFSC_Code' => 'nullable|string|max:20',
            'Account_Type' => 'nullable|in:Savings,Current,OD,CC,NRI,Other',

            // Verification fields
            'PAN_Verified' => 'nullable|boolean',
            'GSTIN_Verified' => 'nullable|boolean',
        ];
    }

    // Custom validation messages (same as before)
    public static function validationMessages()
    {
        return [
            'Lg_Id.required' => 'Ledger is required',
            'Lg_Id.*.exists' => 'Selected ledger does not exist',
            'SL_Code.unique' => 'This sub-ledger code already exists',
            'SL_Name.required' => 'Sub-ledger name is required',
            'SL_Email.email' => 'Please enter a valid email address',
            'PAN_Number.regex' => 'PAN Number must be in valid format (e.g., ABCDE1234F)',
            'GSTIN.regex' => 'GSTIN must be in valid format',
            'Account_Type.in' => 'Account type must be one of: Savings, Current, OD, CC, NRI, Other',
        ];
    }

    // Relationships
    public function ledger()
    {
        return $this->belongsTo(LedgerMaster::class, 'Lg_Id', 'Lg_Id'); // Note: single relation – but we use array in practice
    }

    // Scopes (kept all)
    public function scopeActive($query)
    {
        return $query->where('SL_Status', true);
    }

    public function scopeByLedger($query, $ledgerId)
    {
        return $query->whereJsonContains('Lg_Id', $ledgerId);
    }

    public function scopeSearch($query, $search)
    {
        return $query->where(function ($q) use ($search) {
            $q->where('SL_Code', 'like', "%{$search}%")
                ->orWhere('SL_Name', 'like', "%{$search}%")
                ->orWhere('SL_Cont_Pers', 'like', "%{$search}%")
                ->orWhere('SL_City', 'like', "%{$search}%")
                ->orWhere('SL_Phone', 'like', "%{$search}%")
                ->orWhere('SL_Email', 'like', "%{$search}%")
                ->orWhere('PAN_Number', 'like', "%{$search}%")
                ->orWhere('GSTIN', 'like', "%{$search}%")
                ->orWhere('Account_Number', 'like', "%{$search}%");
        });
    }

    public function scopeWithCreditLimit($query)
    {
        return $query->where('SL_Cr_Lim', '>', 0);
    }

    public function scopeWithDrawingPower($query)
    {
        return $query->where('SL_Draw_Pwr', '>', 0);
    }

    public function scopeHasPAN($query)
    {
        return $query->whereNotNull('PAN_Number')->where('PAN_Verified', true);
    }

    public function scopeHasGSTIN($query)
    {
        return $query->whereNotNull('GSTIN')->where('GSTIN_Verified', true);
    }

    public function scopeIsMSME($query)
    {
        return $query->whereNotNull('MSME_Udyam_Registration_No');
    }

    // Attribute accessors (kept all possible)
    public function getFullAddressAttribute()
    {
        $addressParts = [];
        if ($this->SL_Address) $addressParts[] = $this->SL_Address;
        if ($this->SL_City) $addressParts[] = $this->SL_City;
        if ($this->SL_Pin) $addressParts[] = $this->SL_Pin;
        return implode(', ', $addressParts);
    }

    public function getFormattedCreditLimitAttribute()
    {
        return number_format($this->SL_Cr_Lim, 2);
    }

    public function getFormattedDrawingPowerAttribute()
    {
        return number_format($this->SL_Draw_Pwr, 2);
    }

    public function getStatusTextAttribute()
    {
        return $this->SL_Status ? 'Active' : 'Inactive';
    }

    public function getMaskedPANAttribute()
    {
        if (!$this->PAN_Number) return null;
        return substr($this->PAN_Number, 0, 2) . 'XXXXX' . substr($this->PAN_Number, -2);
    }

    public function getMaskedAccountNumberAttribute()
    {
        if (!$this->Account_Number) return null;
        return 'XXXXXX' . substr($this->Account_Number, -4);
    }

    public function getFormattedPANVerifiedDateAttribute()
    {
        return $this->PAN_Verified_Date ? $this->PAN_Verified_Date->format('d/m/Y') : null;
    }

    public function getFormattedGSTINVerifiedDateAttribute()
    {
        return $this->GSTIN_Verified_Date ? $this->GSTIN_Verified_Date->format('d/m/Y') : null;
    }

    // Business logic methods (kept all)
    public function deactivate()
    {
        $this->update(['SL_Status' => false]);
        return $this;
    }

    public function activate()
    {
        $this->update(['SL_Status' => true]);
        return $this;
    }

    public function getAvailableCredit()
    {
        return max(0, $this->SL_Cr_Lim - $this->SL_Draw_Pwr);
    }

    public function verifyPAN()
    {
        $this->update([
            'PAN_Verified' => true,
            'PAN_Verified_Date' => now()
        ]);
        return $this;
    }

    public function verifyGSTIN()
    {
        $this->update([
            'GSTIN_Verified' => true,
            'GSTIN_Verified_Date' => now()
        ]);
        return $this;
    }

    public function unverifyPAN()
    {
        $this->update([
            'PAN_Verified' => false,
            'PAN_Verified_Date' => null
        ]);
        return $this;
    }

    public function unverifyGSTIN()
    {
        $this->update([
            'GSTIN_Verified' => false,
            'GSTIN_Verified_Date' => null
        ]);
        return $this;
    }

    public function hasCompleteStatutoryDetails()
    {
        return !empty($this->PAN_Number) && !empty($this->GSTIN) && $this->PAN_Verified && $this->GSTIN_Verified;
    }

    public function getComplianceStatusAttribute()
    {
        $score = 0;
        $total = 0;

        if ($this->PAN_Number) {
            $total++;
            if ($this->PAN_Verified) $score++;
        }

        if ($this->GSTIN) {
            $total++;
            if ($this->GSTIN_Verified) $score++;
        }

        return $total > 0 ? round(($score / $total) * 100, 2) : 0;
    }

    public function getComplianceStatusTextAttribute()
    {
        $score = $this->compliance_status;

        if ($score >= 90) return 'Excellent';
        if ($score >= 75) return 'Good';
        if ($score >= 50) return 'Average';
        if ($score >= 25) return 'Poor';
        return 'Very Poor';
    }

    public function getStatutoryDetailsSummaryAttribute()
    {
        $details = [];

        if ($this->PAN_Number) {
            $details['pan'] = [
                'number' => $this->masked_pan,
                'verified' => $this->PAN_Verified,
                'verified_date' => $this->formatted_pan_verified_date
            ];
        }

        if ($this->GSTIN) {
            $details['gstin'] = [
                'number' => $this->GSTIN,
                'verified' => $this->GSTIN_Verified,
                'verified_date' => $this->formatted_gstin_verified_date
            ];
        }

        if ($this->MSME_Udyam_Registration_No) {
            $details['msme'] = [
                'registration_number' => $this->MSME_Udyam_Registration_No
            ];
        }

        if ($this->CIN_LLPIN) {
            $details['company'] = [
                'cin_llpin' => $this->CIN_LLPIN
            ];
        }

        if ($this->TAN) {
            $details['tan'] = [
                'number' => $this->TAN
            ];
        }

        return $details;
    }
}
