<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class WardMaster extends Model
{
    use HasFactory, SoftDeletes;

    protected $table = 'ward_master';
    protected $primaryKey = 'Wrd_Id';

    protected $fillable = [
        'Wrd_Code',
        'Wrd_Name',
        'Zon_Id',
        'is_active'
    ];

    protected $casts = [
        'Wrd_Id' => 'integer',
        'Zon_Id' => 'integer',
        'is_active' => 'boolean'
    ];

    /* ================= VALIDATION RULES ================= */

    public static function createRules()
    {
        return [
            'Wrd_Code' => 'required|string|max:5',
            'Wrd_Name' => 'required|string|max:25',
            'Zon_Id' => 'required|exists:zone_master,Zon_Id'
        ];
    }

    public static function updateRules($id)
    {
        return [
            'Wrd_Code' => 'sometimes|required|string|max:5',
            'Wrd_Name' => 'sometimes|required|string|max:25',
            'Zon_Id' => 'sometimes|required|exists:zone_master,Zon_Id',
            'is_active' => 'sometimes|boolean'
        ];
    }

    public static function validationMessages()
    {
        return [
            'Wrd_Code.required' => 'Ward code is required',
            'Wrd_Code.max' => 'Ward code cannot exceed 5 characters',
            'Wrd_Name.required' => 'Ward name is required',
            'Wrd_Name.max' => 'Ward name cannot exceed 25 characters',
            'Zon_Id.required' => 'Zone is required',
            'Zon_Id.exists' => 'The selected zone does not exist'
        ];
    }

    /* ================= UNIQUE VALIDATION ================= */

    public static function validateUniqueWardCode($code, $zoneId, $excludeId = null)
    {
        $query = self::where('Wrd_Code', $code)
                    ->where('Zon_Id', $zoneId);

        if ($excludeId) {
            $query->where('Wrd_Id', '!=', $excludeId);
        }

        return !$query->exists();
    }

    /* ================= RELATIONSHIPS ================= */

    public function zone()
    {
        return $this->belongsTo(ZoneMaster::class, 'Zon_Id', 'Zon_Id');
    }

    // public function cccs()
    // {
    //     return $this->hasMany(CccMaster::class, 'Wrd_Id', 'Wrd_Id');
    // }

    /* ================= SCOPES ================= */

    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    public function scopeByZone($query, $zoneId)
    {
        return $query->where('Zon_Id', $zoneId);
    }

    public function scopeSearch($query, $search)
    {
        return $query->where(function ($q) use ($search) {
            $q->where('Wrd_Code', 'like', "%{$search}%")
              ->orWhere('Wrd_Name', 'like', "%{$search}%")
              ->orWhereHas('zone', function ($q) use ($search) {
                  $q->where('Zon_Code', 'like', "%{$search}%")
                    ->orWhere('Zon_Name', 'like', "%{$search}%");
              });
        });
    }

    /* ================= ACCESSORS ================= */

    public function getFullWardAttribute()
    {
        return $this->Wrd_Code . ' - ' . $this->Wrd_Name;
    }

    public function getZoneNameAttribute()
    {
        return $this->zone ? $this->zone->Zon_Name : null;
    }

    public function getZoneCodeAttribute()
    {
        return $this->zone ? $this->zone->Zon_Code : null;
    }

    /* ================= BUSINESS LOGIC ================= */

    public function canBeDeleted()
    {
        return $this->cccs()->count() === 0;
    }

    public function deactivate()
    {
        $this->update(['is_active' => false]);
        return $this;
    }

    public function activate()
    {
        $this->update(['is_active' => true]);
        return $this;
    }
}
