<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class ZoneMaster extends Model
{
    use HasFactory, SoftDeletes;

    protected $table = 'zone_master';
    protected $primaryKey = 'Zon_Id';

    protected $fillable = [
        'Zon_Code',
        'Zon_Name',
        'Org_Id'
    ];

    protected $casts = [
        'Zon_Id' => 'integer',
        'Org_Id' => 'integer'
    ];


    public static function createRules()
    {
        return [
            'Zon_Code' => 'required|string|max:5',
            'Zon_Name' => 'required|string|max:25',
            'Org_Id' => 'nullable|integer|exists:organisation_settings,Org_Id'
        ];
    }


    public static function updateRules($id)
    {
        return [
            'Zon_Code' => 'sometimes|required|string|max:5',
            'Zon_Name' => 'sometimes|required|string|max:25',
            'Org_Id' => 'nullable|integer|exists:organisation_settings,Org_Id'
        ];
    }


    public static function validationMessages()
    {
        return [
            'Zon_Code.required' => 'Zon code is required',
            'Zon_Code.max' => 'Zon code cannot exceed 5 characters',
            'Zon_Code.unique' => 'This Zon code already exists for the selected organization',
            'Zon_Name.required' => 'Zon name is required',
            'Zon_Name.max' => 'Zon name cannot exceed 25 characters',
            'Org_Id.exists' => 'The selected organization does not exist'
        ];
    }


    public static function validateUniqueZonCode($ZonCode, $orgId, $excludeId = null)
    {
        $query = self::where('Zon_Code', $ZonCode);

        if ($orgId === null) {
            $query->whereNull('Org_Id');
        } else {
            $query->where('Org_Id', $orgId);
        }

        if ($excludeId) {
            $query->where('Zon_Id', '!=', $excludeId);
        }

        return !$query->exists();
    }


    public function organisation()
    {
        return $this->belongsTo(OrgMaster::class, 'Org_Id', 'Org_Id');
    }

    public function scopeByOrganization($query, $orgId)
    {
        if ($orgId === null) {
            return $query->whereNull('Org_Id');
        }
        return $query->where('Org_Id', $orgId);
    }

    public function scopeWithoutOrganization($query)
    {
        return $query->whereNull('Org_Id');
    }

    public function scopeWithOrganization($query)
    {
        return $query->whereNotNull('Org_Id');
    }

    public function scopeSearch($query, $search)
    {
        return $query->where(function ($q) use ($search) {
            $q->where('Zon_Code', 'like', "%{$search}%")
              ->orWhere('Zon_Name', 'like', "%{$search}%")
              ->orWhereHas('organisation', function ($q) use ($search) {
                  $q->where('Org_Code', 'like', "%{$search}%")
                    ->orWhere('Org_Name', 'like', "%{$search}%");
              });
        });
    }


    public function getFullZonAttribute()
    {
        return $this->Zon_Code . ' - ' . $this->Zon_Name;
    }

    public function getOrganizationNameAttribute()
    {
        return $this->organisation ? $this->organisation->Org_Name : 'Global';
    }

    public function getOrganizationCodeAttribute()
    {
        return $this->organisation ? $this->organisation->Org_Code : null;
    }


    public function isGlobal()
    {
        return $this->Org_Id === null;
    }


    public function canBeDeleted()
    {

        return true;
    }
}
